/*                                         Copyright (c) CMI Productions, 2007
 *****************************************************************************
 *  LedMatrix.c (1.0)
 *
 *  Display rendering functions for the LED matrix
 ****EXTERNAL*****************************************************************
 * transition : routine to initiate a transition
 * transition_engine : routine called by timer to implement transitions
 ****INTERNAL*****************************************************************
 * transition_cut    : "Cut" transition code
 * transition_scroll : "Scroll" transition code
 * transition_wipe   : "Wipe" transition code (just wipe in new pattern)
 *****************************************************************************
	Last change: ETC 2/27/2008 9:05:02 PM
*/

#include <ez80.h>
#include <string.h>

#include "def.h"
#include "globals.h"
#include "asmhelp.h"
#include "tables.h"
#include "transition.h"

/* local global constants and variables (this file only) */

/* internal routines */
void transition_cut    ( void  )  ;
void transition_scroll ( void  )  ;
void transition_wipe   ( UBYTE )  ;


/****************************************************************************
* Routne   : transition
* Gazintas : type: transition type (see ledmatrix.h)
*            direction: transition direction (see ledmatrix.h)
*            lines: which lines to transition (see ledmatrix.h)
*            wait: True to wait until done, false to return without waiting
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array to write to
*            See the file "LED Matrix.txt" for the memory structure
*
* This is the routine to set up a transition, moving data from the standby
* matrix to the active matrix.  This is really just a front routine.  All
* the work is done by ledmatrix_xengine().  This routine just sets up the
* state variables for ledmatrix_xengine to do it's thing and (for now) waits
* for the ledmatrix_xengine code to finish before returning.
****************************************************************************/
void transition( UBYTE type,UBYTE direction,UBYTE lines,UBYTE wait )  
{

   /* if the transition engine is busy, wait */
   while(transition_busy) ;

   /* set up the transition */
   transition_type=type ;
   transition_direction=direction ;
   transition_lines=lines ;

   /* tell the engine to start */
   transition_busy=TRUE ;

   /* if the transition engine is busy, wait */
   if (wait) while(transition_busy) ;
}

/****************************************************************************
* Routne   : transition_engine
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array to write to
*               See the file "LED Matrix.txt" for the memory structure
*            transition_type transition type (see ledmatrix.h)
*            transition_direction transition direction (see ledmatrix.h)
*            transition_lines which lines to transition (see ledmatrix.h)
*
* This is the routine to actually implement the transitions.  It is called
* once every 50ms by the timer interrupt routine.  Data from the "standby"
* display matrix is what will get scrolled in, the three global variables
* above say how to do it, and setting the transition_busy flag is what
* kicks things off.  This is just the handoff routine to call the appropriate
* routine to implement the specific kind of transition.  Those routines do
* thier thing and when done, clear the transition_busy flag to false.
*
****************************************************************************/
void transition_engine ( void )  
{

   if (transition_busy)
   {
      switch (transition_type)
      {
          case TR_CUT    : transition_cut ()       ; break ;
          case TR_SCROLL : transition_scroll ()    ; break ;
          case TR_WIPE1  : transition_wipe (FALSE) ; break ;
          case TR_WIPE2  : transition_wipe (TRUE)  ; break ;
          default        : transition_busy=FALSE   ; break ;
      }
   } 
}

/****************************************************************************
* Routne   : transition_cut
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array to write to
*               See the file "LED Matrix.txt" for the memory structure
*            transition_type transition type (see ledmatrix.h)
*            transition_direction transition direction (see ledmatrix.h)
*            transition_lines which lines to transition (see ledmatrix.h)
*
* This is the routine to implement the "cut" transition.
* It is really simple, we copy data from the standby plane to the active
* plane and declare ourselves done!  Only decision making is which lines
* get copied.
****************************************************************************/
void transition_cut ( void )  
{
   static UBYTE state=0 ; /* Generic state variable */
   UBYTE filler         ; /* filler byte for flash and blink */

   switch (state)
   {
      case 0 : if ((transition_direction==TR_FLASH)||(transition_direction==TR_BLINK))
               {
                  filler=(transition_direction==TR_FLASH)?0xff:0x00 ;

                  if (transition_lines&0x01)
                  {
                     memset(dspmtxa,filler,32) ; /* copy Line 1, Phase A */
                     memset(dspmtxb,filler,32) ; /* copy Line 1, Phase B */
                  }

                  if (transition_lines&0x02)
                  {
                     memset(dspmtxa+32,filler,32) ; /* copy Line 2, Phase A */
                     memset(dspmtxb+32,filler,32) ; /* copy Line 2, Phase B */
                  }

                  state++ ;
                  break ;
               } 
               else /* is TR_FAST, so just copy and exit */
               {
                  if (transition_lines&0x01)
                  {
                     memcpy(dspmtxa,dspstga,32) ; /* copy Line 1, Phase A */
                     memcpy(dspmtxb,dspstgb,32) ; /* copy Line 1, Phase B */
                  }

                  if (transition_lines&0x02)
                  {
                     memcpy(dspmtxa+32,dspstga+32,32) ; /* copy Line 2, Phase A */
                     memcpy(dspmtxb+32,dspstgb+32,32) ; /* copy Line 2, Phase B */
                  }

                  transition_busy=FALSE ;
                  break ;
               } ;

      case 1 : /* only here for TR_FLASH and TR_BLINK, wait a state before data copy */
               state++ ;
               break   ;
      case 2 : /* only here for TR_FLASH and TR_BLINK, wait a state before data copy */
               state++ ;
               break   ;
      case 3 : /* only here for TR_FLASH and TR_BLINK, wait a state before data copy */
               state++ ;
               break   ;
      default: if (transition_lines&0x01)
               {
                  memcpy(dspmtxa,dspstga,32) ; /* copy Line 1, Phase A */
                  memcpy(dspmtxb,dspstgb,32) ; /* copy Line 1, Phase B */
               }

               if (transition_lines&0x02)
               {
                  memcpy(dspmtxa+32,dspstga+32,32) ; /* copy Line 2, Phase A */
                  memcpy(dspmtxb+32,dspstgb+32,32) ; /* copy Line 2, Phase B */
               }

               state=0 ;
               transition_busy=FALSE ;
               break ;
   }
}

/****************************************************************************
* Routne   : transition_scroll
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array to write to
*               See the file "LED Matrix.txt" for the memory structure
*            transition_type transition type (see ledmatrix.h)
*            transition_direction transition direction (see ledmatrix.h)
*            transition_lines which lines to transition (see ledmatrix.h)
*
* This is the routine to implement the "scroll" transition.
* For this we scroll in the data from the standby plane one column at a 
* time (every 50ms) while scrolling the old data "off the opposite edge"
****************************************************************************/
void transition_scroll ( void )  
{
   static UBYTE state=0 ; /* Generic state variable */
   static UBYTE count=0 ; /* generic counter        */

   if (transition_direction==TR_LEFT)
   {
      /* For shift left we treat each line as 32 bits of active display    */
      /* concatentaed with 32 bits of standby display and shift everything */
      /* one pixel left once every 50ms.  This code shifts everything one  */
      /* pixel left, so it just gets called 32 times before being done.    */
      /* The assembly routine xshiftl decides which lines to shift.        */
      switch (state)
      {
         case 0 : count=0 ;
                  state++ ;
                  break   ;
         case 1 : xshiftl(transition_lines) ;
                  count++ ;
                  if (count==32) state ++ ;
                  break ;
         default: transition_busy=FALSE ;
                  state=0               ;
                  break                 ;
      }
   }
   else if (transition_direction==TR_RIGHT)
   {
      /* For shift right we treat each line as 32 bits of active display    */
      /* concatentaed with 32 bits of standby display and shift everything  */
      /* one pixel right once every 50ms.  This code shifts everything one  */
      /* pixel right, so it just gets called 32 times before being done.    */
      /* The assembly routine xshiftr decides which lines to shift.         */
      switch (state)
      {
         case 0 : count=0 ;
                  state++ ;
                  break   ;
         case 1 : xshiftr(transition_lines) ;
                  count++ ;
                  if (count==32) state ++ ;
                  break ;
         default: transition_busy=FALSE ;
                  state=0               ;
                  break                 ;
      }
   }
   else if (transition_direction==TR_DOWN)
   {
      switch (state)
      {
         case 0 : count=0 ;
                  state++ ;
                  break   ;
         case 1 : if (transition_lines==TR_LINE1)
                  {
                     /* copy lines 2-8 down one line (both planes in active matrix) */
                     memmove(dspmtxa+4,dspmtxa,28) ;
                     memmove(dspmtxb+4,dspmtxb,28) ;

                     /* copy line 8 from the staging area to line 1 of the active matrix */
                     memmove(dspmtxa,dspstga+28,4) ;
                     memmove(dspmtxb,dspstgb+28,4) ;

                     /* copy lines 2-8 down one line (both planes in staging plane) */
                     memmove(dspstga+4,dspstga,28) ;
                     memmove(dspstgb+4,dspstgb,28) ;

                     /* vertical scrolling every 50ms is too fast, so wait a state before next */
                     state++ ;
                     /* if we're done, increment state twice to get to end */
                     count++ ;
                     if(count==8) state++ ;
                  }
                  else if (transition_lines==TR_LINE2)
                  {
                     /* copy lines 9-16 down one line (both planes in active matrix) */
                     memmove(dspmtxa+36,dspmtxa+32,28) ;
                     memmove(dspmtxb+36,dspmtxb+32,28) ;

                     /* copy line 16 from the staging area to line 9 of the active matrix */
                     memmove(dspmtxa+32,dspstga+60,4) ;
                     memmove(dspmtxb+32,dspstgb+60,4) ;

                     /* copy lines 9-16 down one line (both planes in staging plane) */
                     memmove(dspstga+36,dspstga+32,28) ;
                     memmove(dspstgb+36,dspstgb+32,28) ;

                     /* vertical scrolling every 50ms is too fast, so wait a state before next */
                     state++ ;
                     /* if we're done, increment state twice to get to end */
                     count++ ;
                     if(count==8) state++ ;
                  }
                  else if (transition_lines==TR_LINE12)
                  {
                     /* copy lines 2-16 down one line (both planes in active matrix) */
                     memmove(dspmtxa+4,dspmtxa,60) ;
                     memmove(dspmtxb+4,dspmtxb,60) ;

                     /* copy line 16 from the staging area to line 1 of the active matrix */
                     memmove(dspmtxa,dspstga+60,4) ;
                     memmove(dspmtxb,dspstgb+60,4) ;

                     /* copy lines 2-8 down one line (both planes in staging plane) */
                     memmove(dspstga+4,dspstga,60) ;
                     memmove(dspstgb+4,dspstgb,60) ;

                     /* vertical scrolling every 50ms is too fast, so wait a state before next */
                     state++ ;
                     /* if we're done, increment state twice to get to end */
                     count++ ;
                     if(count==16) state++ ;
                  }
                  break ;
         case 2 : state-- ; /* simply a delay state, go back to state 1 after 50ms */
                  break                 ;
         default: transition_busy=FALSE ;
                  state=0               ;
                  break                 ;
      }
   }
   else if (transition_direction==TR_UP)
   {
      switch (state)
      {
         case 0 : count=0 ;
                  state++ ;
                  break   ;
         case 1 : if (transition_lines==TR_LINE1)
                  {
                     /* copy lines 1-7 up one line (both planes in active matrix) */
                     memmove(dspmtxa,dspmtxa+4,28) ;
                     memmove(dspmtxb,dspmtxb+4,28) ;

                     /* copy line 1 from the staging area to line 8 of the active matrix */
                     memmove(dspmtxa+28,dspstga,4) ;
                     memmove(dspmtxb+28,dspstgb,4) ;

                     /* copy lines 1-7 up one line (both planes in staging plane) */
                     memmove(dspstga,dspstga+4,28) ;
                     memmove(dspstgb,dspstgb+4,28) ;

                     /* vertical scrolling every 50ms is too fast, so wait a state before next */
                     state++ ;
                     /* if we're done, increment state twice to get to end */
                     count++ ;
                     if(count==8) state++ ;
                  }
                  else if (transition_lines==TR_LINE2)
                  {
                     /* copy lines 8-15 up one line (both planes in active matrix) */
                     memmove(dspmtxa+32,dspmtxa+36,28) ;
                     memmove(dspmtxb+32,dspmtxb+36,28) ;

                     /* copy line 8 from the staging area to line 16 of the active matrix */
                     memmove(dspmtxa+60,dspstga+32,4) ;
                     memmove(dspmtxb+60,dspstgb+32,4) ;

                     /* copy lines 8-15 up one line (both planes in staging plane) */
                     memmove(dspstga+32,dspstga+36,28) ;
                     memmove(dspstgb+32,dspstgb+36,28) ;

                     /* vertical scrolling every 50ms is too fast, so wait a state before next */
                     state++ ;
                     /* if we're done, increment state twice to get to end */
                     count++ ;
                     if(count==8) state++ ;
                  }
                  else if (transition_lines==TR_LINE12)
                  {
                     /* copy lines 2-16 up one line (both planes in active matrix) */
                     memmove(dspmtxa,dspmtxa+4,60) ;
                     memmove(dspmtxb,dspmtxb+4,60) ;

                     /* copy line 1 from the staging area to line 16 of the active matrix */
                     memmove(dspmtxa+60,dspstga,4) ;
                     memmove(dspmtxb+60,dspstgb,4) ;

                     /* copy lines 2-8 down one line (both planes in staging plane) */
                     memmove(dspstga,dspstga+4,60) ;
                     memmove(dspstgb,dspstgb+4,60) ;

                     /* vertical scrolling every 50ms is too fast, so wait a state before next */
                     state++ ;
                     /* if we're done, increment state twice to get to end */
                     count++ ;
                     if(count==16) state++ ;
                  }
                  break ;
         case 2 : state-- ; /* simply a delay state, go back to state 1 after 50ms */
                  break                 ;
         default: transition_busy=FALSE ;
                  state=0               ;
                  break                 ;
      }
   }

}

/****************************************************************************
* Routne   : transition_wipe
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array to write to
*               See the file "LED Matrix.txt" for the memory structure
*            transition_type transition type (see ledmatrix.h)
*            transition_direction transition direction (see ledmatrix.h)
*            transition_lines which lines to transition (see ledmatrix.h)
*
* This is the routine to implement the "wipe" transition.
* For this we just start replacing old pixels with new pixels one row/colum
* at a time.  The difference between this and wipe2 is that for wipe2 I have
* a row/column of LEDs on at the wipe point.
****************************************************************************/
void transition_wipe ( UBYTE bar )
{
   static UBYTE state=0   ; /* Generic state variable   */
   static BYTE count=0    ; /* generic counter          */
   static BYTE dir,last   ; /* counter direction        */
   static UBYTE start,end ; /* starting and ending rows */
   UBYTE idx,tmp,i        ; /* for wipes                */
   UBYTE mask1,mask2      ; /* for wipes                */
   
   /* turns out to be pretty easy to combine left and right wipe routines! */
   if ((transition_direction==TR_LEFT)||(transition_direction==TR_RIGHT))
   {
      switch (state)
      {
         case 0 : /* set up starting and ending counts as well as direction */
                  if (transition_direction==TR_LEFT)
                  {
                     count = 31 ;
                     dir   = -1 ;
                     last  =  0 ;
                  }
                  else
                  {
                     count = 0  ;
                     dir   = 1  ;
                     last  = 31 ;
                  }

                  /* determine range of lines that get wiped */
                  switch(transition_lines)
                  {
                     case TR_LINE1  : start=0 ; end=8  ; break ;
                     case TR_LINE2  : start=8 ; end=16 ; break ;
                     case TR_LINE12 : start=0 ; end=16 ; break ;
                     default        : start=0 ; end=8  ; break ;
                  } ;
                  state++ ;
                  break   ;
         case 1 : /* mask values are constant for a column, so compute once */
                  mask1=oneofeight[count&0x07] ;
                  mask2=oneofeight[(count+dir)&0x07] ;

                  /* now go copy a colum of pixels.  Count depends on lines */
                  for (i=start;i<end;i++)
                  {
                     idx=(count/8)+i*4         ; /* index into active and standby array */
                     tmp=dspmtxa[idx]&(~mask1) ; /* get bit from active array and delete mask bit */
                     tmp|=dspstgb[idx]&mask1   ; /* merge in bit from staging array */
                     dspmtxa[idx]=tmp          ; /* write it back */

                     tmp=dspmtxb[idx]&(~mask1) ; /* do same for phase B */
                     tmp|=dspstgb[idx]&mask1   ;
                     dspmtxb[idx]=tmp          ;

                     /* if this is a wipe2 (with bar) I write a bar to column */
                     /* "before" the one just copied.  This is cool as on the */
                     /* next loop, it will get over-written by staging array  */
                     /* data.  But, don't do it for the last column!          */
                     if (bar&&(count!=last))
                     {
                        idx=((count+dir)/8)+i*4 ;
                        dspmtxa[idx]|=mask2 ;
                        dspmtxb[idx]|=mask2 ;
                     }
                  } ;

                  if (count==last) state ++ ;
                  count+=dir ;
                  break ;
         default: transition_busy=FALSE ;
                  state=0               ;
                  break                 ;
      }
   }
   else if ((transition_direction==TR_DOWN)||(transition_direction==TR_UP))
   {
      switch (state)
      {
         case 0 : /* determine the starting lines, ending lines, and direction */
                  /* for the wipe.                                             */
                  if (transition_direction==TR_DOWN)
                  {
                     dir = 1 ;
                     switch(transition_lines)
                     {
                        case TR_LINE1  : count=0 ; end=7  ; break ;
                        case TR_LINE2  : count=8 ; end=15 ; break ;
                        case TR_LINE12 : count=0 ; end=15 ; break ;
                        default        : count=0 ; end=7  ; break ;
                     } ;
                  }
                  else
                  {
                     dir = -1 ;
                     switch(transition_lines)
                     {
                        case TR_LINE1  : count=7  ; end=0 ; break ;
                        case TR_LINE2  : count=15 ; end=8 ; break ;
                        case TR_LINE12 : count=15 ; end=0 ; break ;
                        default        : count=7  ; end=0 ; break ;
                     } ;
                  }

                  state++ ;
                  break   ;
         case 1 : /* for the loop, simply copy lines of pixels from the staging */
                  /* array to the active array (both phases)                    */
                  idx=count*4 ;
                  memmove(dspmtxa+idx,dspstga+idx,4) ;
                  memmove(dspmtxb+idx,dspstgb+idx,4) ;

                  /* if this is a wipe2 (with bar) I write a bar to row    */
                  /* "before" the one just copied.  This is cool as on the */
                  /* next loop, it will get over-written by staging array  */
                  /* data.  But, don't do it for the last row!             */
                  if (bar&&(count!=end))
                  {
                     idx+=dir*4 ;
                     memset(dspmtxa+idx,0xff,4) ;
                     memset(dspmtxb+idx,0xff,4) ;
                  }

                  if (count==end) state ++ ;
                  count+=dir ;
                  break ;
         default: transition_busy=FALSE ;
                  state=0               ;
                  break                 ;
      }
   }
}
