/*                                         Copyright (c) CMI Productions, 2007
 *****************************************************************************
 *  time.c
 *
 *  Time functions
 ****EXTERNAL*****************************************************************
 * time_init : Initialize the time to 12:00 1/1/07
 * time_set  : Set the time manually
 * time_get  : Get the time from WWVB (really just watch)
 * time_inc  : Increment time in 1/60ths of a second
 * time_dsp  : Write out the time to the LED matrix
 ****INTERNAL*****************************************************************
 * time_outer    : Write to the outer LED ring
 * time_inner    : Write to the inner LED ring
 * days_in_month : The number of days in the month
 *****************************************************************************
*/

#include <ez80.h>
#include <string.h>

#include "def.h"
#include "globals.h"
#include "time.h"
#include "LedMatrix.h"
#include "tables.h"
#include "timer.h"
#include "transition.h"
#include "gpio.h"
#include "wwvb.h"

/* local global constants and variables (this file only) */

/* internal routines */
UBYTE days_in_month (UBYTE, UBYTE) ;
void  time_outer (UBYTE,UBYTE) ;
void  time_inner (UBYTE,UBYTE,UBYTE) ;

/****************************************************************************
* Routne   : time_init
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : ssecs,secs,mins,hrs,day,month,year
*
* This routine just initializes these to a known state.  Later routines will
* hopefully set it to the "correct" state using WWVB or NTP.
****************************************************************************/
void time_init ( void )
{
   ssecs  = 0     ; /* Sub-seconds (1/60ths)   */
   secs   = 0     ; /* Seconds                 */
   mins   = 0     ; /* Minutes                 */
   hrs    = 0     ; /* Hours                   */
   day    = 1     ; /* Day                     */
   month  = 1     ; /* Month                   */
   year   = 7     ; /* Year                    */
   newmin = TRUE  ; /* new minute flag         */
   thold  = FALSE ; /* allow time to increment */
}

/****************************************************************************
* Routne   : time_set
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : ssecs,secs,mins,hrs,day,month,year
*
* This routine is for setting the time manually.
****************************************************************************/
void time_set ( void )
{
   UBYTE state=0 ; /* what is being set */

   /* stop time from incrementing while we are setting it and */
   /* reset sub-seconds and seconds.                          */
   thold = TRUE ;
   secs  = 0    ;
   ssecs = 0    ;

   
   while (state<5) /* five stats (hrs, mins, month, day, year) */
   {
      /* Dismiss the current key so we react to the next keypress. */
      curkey = KEY_NONE ;

      /* first write out the curren time */
	  time_dsp (SHOWTIME_PLAIN) ;
	  switch (state)
	  {
	     case 0 : ledmatrix_wrtime (SHOWTIME_SETHRS,TRT_NORMAL)   ; break ;
	     case 1 : ledmatrix_wrtime (SHOWTIME_SETMINS,TRT_NORMAL)  ; break ;
		 case 2 : ledmatrix_wrtime (SHOWTIME_SETMONTH,TRT_NORMAL) ; break ;
		 case 3 : ledmatrix_wrtime (SHOWTIME_SETDAY,TRT_NORMAL)   ; break ;
		 case 4 : ledmatrix_wryear (SHOWTIME_SETYEAR)             ; break ;
		 default :                                                ; break ;
	  } ;

      /* now wait for a keypress */
	  while (curkey==KEY_NONE) ;

      if (curkey==KEY_MENU)       /* if is MENU, increment the state */
	  {
	     state++ ;
	  }
      else if (curkey==KEY_UP)  /* if key is UP, increment the time value */
	  {
         switch (state)
	     {
	        case 0 : hrs++ ;
		             if (hrs>23) hrs=0 ;
		             break ;
	        case 1 : mins++ ;
                     if (mins>59) mins=0 ;
                     break ;
	        case 2 : month++ ;
                     if (month>12) month=1 ;
                     break ;
	        case 3 : day++ ;
                     if (day>days_in_month(month,year)) day=1 ;
                     break ;
	        case 4 : year++ ;
                     if (year>99) year=0 ;
                     break ;
            default : break ;
		 }
	  }
      else if (curkey==KEY_DOWN)  /* if key is DOWN, decrement the time value */
	  {
         /* I rely on unsigned math here to detect roll-over, so it looks a little weird */
         switch (state)
	     {
	        case 0 : hrs-- ;
		             if (hrs>23) hrs=23 ;
		             break ;
	        case 1 : mins-- ;
                     if (mins>59) mins=59 ;
                     break ;
	        case 2 : month-- ;
                     if (month==0) month=12 ;
                     break ;
	        case 3 : day-- ;
                     if (day==0) day=days_in_month(month,year) ;
                     break ;
	        case 4 : year-- ;
                     if (year>99) year=99 ;
                     break ;
            default : break ;
		 }
	  }
   } ; /* exit here if they hit menu for the last time */
  
   /* Dismiss the final menu key and let time increment before going back to main */
   curkey = KEY_NONE ;
   thold = FALSE ;
}

/****************************************************************************
* Routne   : time_get
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : None directly
*
* This routine just monitors the WWVB state machine status and displays it.
* This is for the first-time acquisition of the time.  All other times it is
* done in the background while normal time functions are going on.
****************************************************************************/
void time_get ( void )
{
   UBYTE last_state=WWVB_IDLE ; /* last state so we can catch transitions */
   UBYTE last_bit=0           ; /* last bit received                      */
   UBYTE gsym=0               ; /* local copy of symbol received          */

   /* stop regular time and fire off the WWVB state machine for starters  */
   wwvb_go() ;

   ledmatrix_clear(ACTIVE_PLANE) ;

   /* debug code block #2 */
   while (0) ; //(wwvb_stat(WWVB_RSTATE)!=WWVB_IDLE)
   {
      gsym=wwvb_stat(WWVB_SYM) ;
      if (gsym!=0) /* if we got a new symbol */
	  {
         ledmatrix_cline(ACTIVE_PLANE,1) ;
         switch (gsym)
		 {
		    case 1 : ledmatrix_putc(10,0,'1',PHASEAB,ACTIVE_PLANE) ; break ;
		    case 2 : ledmatrix_putc(10,0,'0',PHASEAB,ACTIVE_PLANE) ; break ;
		    case 3 : ledmatrix_putc(10,0,'P',PHASEAB,ACTIVE_PLANE) ; break ;
		    case 4 : ledmatrix_putc(10,0,'E',PHASEAB,ACTIVE_PLANE) ;
                     dspmtxa[32]=dspmtxb[32]=wwvb_stat(WWVB_CNT)   ;
			         break ;
		    default: ledmatrix_putc(10,0,'?',PHASEAB,ACTIVE_PLANE) ; break ;
		 }
	  }
   }

   /* this is debug code, so if we get here, return without running normal code below */
   //return ;


   /* do this until we're no longer idle */
   while (wwvb_stat(WWVB_RSTATE)!=WWVB_IDLE)
   {
      /* display if we entered the SYNC state */
	  if ((wwvb_stat(WWVB_RSTATE)==WWVB_SYNC1)&&(last_state!=WWVB_SYNC1))
	  {
         ledmatrix_clear(ACTIVE_PLANE) ;
         ledmatrix_puts(3,0,(UBYTE*)"WWVB",PHASEAB,ACTIVE_PLANE) ;
         ledmatrix_puts(3,8,(UBYTE*)"SYNC",PHASEAB,ACTIVE_PLANE) ;
		 last_state=WWVB_SYNC1 ;
		 last_bit=0 ;
	  }
      
      /* display if we entered the RX1 state */
	  if ((wwvb_stat(WWVB_RSTATE)==WWVB_GDATA)&&(last_state!=WWVB_GDATA))
	  {
         ledmatrix_clear(ACTIVE_PLANE) ;
         ledmatrix_puts(3,0,(UBYTE*)"WWVB",PHASEAB,ACTIVE_PLANE) ;
         ledmatrix_puts(7,8,(UBYTE*)"RX1" ,PHASEAB,ACTIVE_PLANE) ;
		 last_state=WWVB_GDATA ;
		 time_outer(0,PHASEAB) ;
	  }

      /* if we are getting data, show the bit count */
	  if ((last_state==WWVB_GDATA)&&(wwvb_stat(WWVB_RBIT)!=last_bit))
      {
		 last_bit=wwvb_stat(WWVB_RBIT) ;
		 time_outer(last_bit,PHASEAB) ;
	  }
   }
}

/****************************************************************************
* Routne   : time_inc
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : ssecs,secs,mins,hrs,day,month,year
*
* This routine increments the time by 1/60th of a second.  Pretty normal 
* stuff except for DST transitions.
****************************************************************************/
void time_inc ( void )
{
   /* if time is in hold, do nuthin'  */
   if (thold) return ;

   ssecs++ ; /* Sub-seconds (1/60ths) */
   if (ssecs<60) return ;
   
   ssecs=0 ;
   secs++  ; /* Seconds (normally 60, but 61 for a leap-second) */
   if (secs<nsecs) return ;

   secs=0      ;
   mins++      ; /* Minutes           */
   newmin=TRUE ; /* flag new minute   */
   if (mins<60) return ;

   mins=0 ;
   hrs++   ; /* Hours                 */

   /* For DST transitions I look at the hour.  If we just got to 2am */
   /* local I look at the dstflag variable and do what it says.      */
   if (hrs==2)
   {
     switch (dstflag)
	 {
	    case TO_DST : /* Spring Forward */
		              hrs++ ;
					  break ;
		case TO_ST  : /* Fall Back */
		              hrs-- ;
					  break ;
		default     : /* do nothing */
		              break ;
      }
      /* whatever the direction, don't do it again */
	  dstflag=NOTHING ;
   }

   if (hrs<24) return ;
   
   /* if local midnight, go get the time again */
   wwvb_go() ;

   hrs=0 ;
   day++   ; /* Day                   */
   if (day<=days_in_month(month,year)) return ;

   day=1 ;
   month++ ; /* Month                 */
   if (month<13) return ;
   
   month=1 ;
   year++ ; /* Year                   */
   if (year<100) return ;
   year=00 ;
}

/****************************************************************************
* Routne   : time_dsp
* Gazintas : mode : display mode
* IOs      : None
* Returns  : Nothing
* Globals  : ssecs,secs,mins,hrs
*
* This routine writes out the time to the two rings of LEDs.  Hence it uses
* Hours, Minuts, seconds, and optionally sub-seconds
****************************************************************************/
void time_dsp (UBYTE mode)
{
   UBYTE i ;

   if (mode!=SHOWTIME_OFF)
   {
      /* if mode is SSSEQ, re-assign mode based on minute */
	  if (mode==SHOWTIME_SSSEQ) mode=(mins%5)+1 ;
   
   /* first clear the array */
      memset(clkmtxa,0,16) ;
      memset(clkmtxb,0,16) ;

      /* set the hours (also sets the tick marks) */
      time_inner(mins,hrs,PHASEA) ;   

      /* if we are actively getting the time, blink the noon tick mark */
      if (wwvb_stat(WWVB_RSTATE)!=WWVB_IDLE)
	  {
         clkmtxa[8]=(clkmtxa[8]&0xfe)|wwvb_led ;
         clkmtxb[8]=(clkmtxb[8]&0xfe)|wwvb_led ;
	  }

      /* set the LEDs for minutes */
      time_outer(mins,PHASEB) ;

      /* see if this is a leap-second */
	  if (secs==60)
	  {
	     /* if so, flash outside lights big-time */
         memset(clkmtxa,0x55,8) ;
         memset(clkmtxb,0xaa,8) ;
	  }
	  else /* otherwise normal processing */
      {
         /* set the LEDs for seconds unless in mode 4 (taken care of later) */
         if (mode!=SHOWTIME_SSMODE4) time_outer(secs,STEADY) ;

         /* if mode=1 or 2, also set sub-seconds */
         if ((mode==SHOWTIME_SSMODE1)||(mode==SHOWTIME_SSMODE2)) time_outer(ssecs,STEADY) ;

         /* if mode=2, also set negative sub-seconds */
         if (mode==SHOWTIME_SSMODE2) time_outer(59-ssecs,STEADY) ;

         /* if mode=3, alternating back and forth */
         if (mode==SHOWTIME_SSMODE3)
	     {
	        if (secs&0x01)
		    {
		       time_outer(59-ssecs,STEADY) ;
		    }
		    else
		    {
		       time_outer(ssecs,STEADY) ;
		    }
         }

         /* if mode=4, just show bargraph seconds */
         if (mode==SHOWTIME_SSMODE4)
	     {
            for (i=0;i<secs/8;i++)
		    {
		       clkmtxa[i]^=0xff ;
		       clkmtxb[i]^=0xff ;
		    } ;

		    clkmtxa[i]^=barcode[secs%8] ;
		    clkmtxb[i]^=barcode[secs%8] ;
         }
      }
   }

}


/****************************************************************************
* Routne   : days_in_month
* Gazintas : month, year : The month to check
* IOs      : None
* Returns  : days in the specified month
* Globals  : days_per_month
*
* This routine returns the number of days for the specified month.  I have
* an array for each month in the year ten do some post-processing to figure
* out leap years.  Should be good forever.
****************************************************************************/
UBYTE days_in_month (UBYTE month, UBYTE year)
{
   UBYTE answer ;
   
   /* months are 1-12, array indecees are 0-11! */
   answer=days_per_month[month-1] ;

   if (month==2)
   {
      /* compute leap-year cycles default from above is 28 days */
      if ((year%4)==0)   answer=29 ; /* Year divisible by 4 = leap year...         */
      if ((year%100)==0) answer=28 ; /* ...unless divisible by 100, then not       */
      /* I'd normally do the 400-year check, but I only care about 2000-2099       */
//    if ((year%400)==0) answer=29 ; /* ...unless also divisible by 400, then yes! */
   }
   
   return (answer) ;
}

/****************************************************************************
* Routne   : time_outer
* Gazintas : tick : minute/second to light
* IOs      : None
* Returns  : Nothing
* Globals  : clkmtx : Matrix for rings of LEDs
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine lights the LED corrosponding to the minute/second provided.
* LEDs are cumulatively lit.  I do some bounds checking but do not return
* error codes.
* Display memory is set up such that there is a bit per second/minute, so
* simply "light" the nth bit in the sequence, byte # seconds/8 and bit is
* seconds modulo 8.
****************************************************************************/
void time_outer (UBYTE tick,UBYTE phase)
{
   /* some quick bounds checking */
   if (tick>59) return ;

   /* set the bit */
   if (phase&0x01) clkmtxa[tick>>3]|=oneofeight[tick&0x07] ;
   if (phase&0x02) clkmtxb[tick>>3]|=oneofeight[tick&0x07] ;
}

/****************************************************************************
* Routne   : time_inner
* Gazintas : mins, hrs
* IOs      : None
* Returns  : Nothing
* Globals  : clkmtx : Matrix for rings of LEDs
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine lights the LED corrosponding to the hour provided.  Since 
* the inner ring supports quarter-hours, minutes need to be provided too.
* LEDs are cumulatively lit.  I do some bounds checking but do not return
* error codes.  The hours circl also contains the hours "tick marks", so 
* it is easier to simply use a lookup table.
****************************************************************************/
void time_inner (UBYTE mins, UBYTE hrs,UBYTE phase)
{
   UBYTE index ;

   /* some bounds checking */
   if (mins>59) return ;
   if (hrs>23) return ;

   /* fill in the tick marks first */
   memcpy(clkmtxa+8,ticks,8) ;
   memcpy(clkmtxb+8,ticks,8) ;

   /* index is a 0-47 value for each 15 minuts of a 12-hour day. */
   index=(hrs%12)*4+mins/15 ;

   /* now go fill in the LED based on the ticks array */
   if (phase&0x01) clkmtxa[hrled[index][0]+8] |= hrled[index][1] ;
   if (phase&0x02) clkmtxb[hrled[index][0]+8] |= hrled[index][1] ;
}

