/*                                         Copyright (c) CMI Productions, 2007
 *****************************************************************************
 *  GPIO.c (1.0)
 *
 *  General purpose IO routines
 ****EXTERNAL*****************************************************************
 * gpio_init     : initialize the display "engine"
 * gpio_rdkeys   : Read and process buttons
 * gpio_nextkey  : Waits for key unpress then next press
 * gpio_ena_wwvb : Enable the WWVB receiver
 * gpio_get_wwvb : Get the state of the WWVB receiver
 *
 ****INTERNAL*****************************************************************
 * None
 *****************************************************************************
*/

#include <ez80.h>

#include "def.h"
#include "globals.h"
#include "gpio.h"
#include "tables.h"

/****************************************************************************
* Routne   : gpio_init
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : None (directly)
*
* This routine sets up the GPIO pins and any other miscellaneous hardware
* PORT A: PA[0] : GPO '595 RCK (clocks shift register to output register)
*         PA[1] : GPO '595 SRCLR* (clears the shift register (not needed))
*	      PA[2] : GPO '595 SRCK (shift register clock)
*         PA[3] : ALT '595 G Timer OC3 (LED cathode output enable)
*	      PA[4] : GPI WWVB signal (active high pulse)
*	      PA[5] : GPO WWVB enable (active low)
*	      PA[6] : GPO Unused
*	      PA[7] : GPO Unused (debug flag for ISR active)
* PORT B: PB[0:7] : GPO ROW drivers 0-7
* PORT C: PC[0:7] : GPO Data to shift registers
* PORT D: PD[0] : ALT UART Tx
*         PD[1] : ALT UART Rx
*         PD[2] : GPI Up switch
*         PD[3] : GPI Down switch
*         PD[4] : GPI Enter switch
*         PD[5] : GPI Menu switch
*         PD[6] : GPO Unused
*         PD[7] : GPO Unused
*
****************************************************************************/
void gpio_init ( void)
{
   /* initialize port A (will go into an init routine later) */
   /* All are GPIO set low except for bit 3 which is alternate function */
   PA_DR  =0x20 ;
   PA_DDR =0x18 ;
   PA_ALT1=0x00 ;
   PA_ALT2=0x08 ;

   /* Port B is all outputs */
   PB_DR  =0x00 ;
   PB_DDR =0x00 ;
   PB_ALT1=0x00 ;
   PB_ALT2=0x00 ;

   /* Port C is all outputs */
   PC_DR  =0x00 ;
   PC_DDR =0x00 ;
   PC_ALT1=0x00 ;
   PC_ALT2=0x00 ;

   /* Port D bits 0 and 1 are special funcions (UART) */
   /* Port D bits 2 to  5 are inputs                  */
   /* Port D bits 6 and 7 are unused (outputs)        */
   PD_DR  =0x00 ;
   PD_DDR =0x3f ;
   PD_ALT1=0x00 ;
   PD_ALT2=0x03 ;
}

/****************************************************************************
* Routne   : gpio_rdkeys
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : curkey
*
* This routine reads the four IO pins associated with the four buttons and
* processes them.  This routine gets called by the sequencer once every
* 50ms so I get to do things like debouncing as well as auto-repeat.
* A valid key is flagged by a non-zero value of the global variable curkey.
* As soon as the application reads the key, it should set the value back to
* zero.
****************************************************************************/
void gpio_rdkeys ( void )
{
   UBYTE rdkey,tmp ;
   static UBYTE prevkey = KEY_NONE ;
   static UBYTE keycount = 0 ;

   /* read in the raw port value and translate to a keycode */
   tmp=PD_DR ;
   rdkey=keytable[(tmp>>2)&0x0f] ;

   /* if current value isn't same as last default to no key */
   if (rdkey!=prevkey)
   {
      prevkey=rdkey ; 
      rdkey=KEY_NONE ;
   }

   /* Now do the repeat stuff which is only for UP and DOWN.                */
   /* Basic idea is to count how many times in succession I see the key     */
   /* being pressed.  At a count of 1 (first press) I set the curkey so the */
   /* app can see it.  For successive counts 2-19 I do nothing.  I peg at a */
   /* count of 20 and once there, set curkey every time I still see it      */
   /* pressed.  That gives a delay of roughly 1 second and a repeat time of */
   /* 50ms.  Seeing a rdkey value of no key resets the counter              */
   if (rdkey==KEY_NONE)
   {
      keycount = 0 ;
	  curkey=KEY_NONE ;
   }
   else
   {
	  if (keycount<20) keycount++ ;
	  if (keycount==1)
	  {
	     curkey=rdkey ;
	  }
	  if ((keycount==20)&&((rdkey==KEY_UP)||(rdkey==KEY_DOWN)))
	  {
	     curkey=rdkey ;
	  }
   }
}

/****************************************************************************
* Routne   : gpio_nextkey
* Gazintas : None
* IOs      : None
* Returns  : key
* Globals  : None
*
* This routine waits for there to be no keys pressed, then returns as soon
* as the next key is pressed.
****************************************************************************/
UBYTE gpio_nextkey ( void )
{
   UBYTE key ; /* the keypress to return */

   /* wait for no key to be pressed */
   while (curkey!=KEY_NONE) ;

   /* now wait for a key to be pressed */
   while ((key=curkey)==KEY_NONE) ;

   /* return the value ! */
   return (key) ;
}

/****************************************************************************
* Routne   : gpio_nokey
* Gazintas : None
* IOs      : None
* Returns  : key
* Globals  : None
*
* This routine waits for there to be no keys pressed.
****************************************************************************/
void gpio_nokey ( void )
{
   /* wait for no key to be pressed */
   while (curkey!=KEY_NONE) ;
}

/****************************************************************************
* Routne   : gpio_ena_wwvb
* Gazintas : enable
* IOs      : None
* Returns  : Nothing
* Globals  : dhold
*
* This routine enables or disables the WWVB radio.  It needs to be disabled
* when not in use, otherwise it can take minutes to "warm up".  The radio
* is enabled by having GPIO port PA5 set to 0.  It is disabled by having
* that port be a 1. Note that I don't actually set the port value here, just
* the sempahore.  The semaphore is picked up by the display refresh routines
* within a milli-second
****************************************************************************/
void gpio_ena_wwvb (UBYTE enable)
{
   wwv_pa = enable?0x00:0x20 ;
}


/****************************************************************************
* Routne   : gpio_get_wwvb
* Gazintas : None
* IOs      : None
* Returns  : 1 if Pulse active, 0 otherwise.
* Globals  : None
*
* This routine gets the state of WWVB signal.  True if the pulse is active
* IO port high), false if it is the inactive time.  This is kind of an abuse
* of a routine call.
****************************************************************************/
UBYTE gpio_get_wwvb (void)
{
   return ((PA_DR&0x10)?1:0) ;
}

