; =============================================================================
; These are the assembly language helper routines.  I use them for anything
; time critical where regular C isn't efficient enough.  There is a pattern,
; I can't get C to do multi-byte shifts efficiently, so most routines do seem
; to involve shifts.
; =============================================================================
;	send_clk: write out a byte of ring data to the shifty registers
;   wrchr: Copies 5x8 character to display memory
;   bfshift: Do a right-shift on a byte array (also fills in LSBit to 1)
;   xshiftl: One-pixel shift left from standby array to active array 
;   xshiftr: One-pixel shift right from standby array to active array 
;   aldir: easy access to LDIR instruction
;   alddr: easy access to LDIR instruction
;   rlmem: Bit shift left of memory array
; =============================================================================
; Things to remember:
; 1) Only IX and SP need to be preserved by assembly code
; 2) Stick to the three (PUSH, LD, ADD) instructions at the start
; 3) Stick to the three (LD, POP, RET) instructions at the end.
; 4) Paramebter #1 is at (ix+3), #2 at (ix+6)... BUT
;    Since we push IX on the stack, Parameter #1 is at (ix+6), #2 at (ix+9)...
; 5) Parameters are pushed last-parameter-first
; 6) Help topic is "Calling Assembly from C (eZ80)"
; =============================================================================
		.assume ADL=1
INCLUDE		"ez80f91.inc"

; Stuff in this file that needs to be found globally
XDEF	_send_clk
XDEF    _wrchr
XDEF    _bfshift
XDEF    _xshiftl
XDEF    _xshiftr
XDEF	_aldir
XDEF	_alddr
XDEF	_rlmem

; Global stuff that needs to be found locally
XREF	_dspmtxa, _dspmtxb
XREF    _dspstga, _dspstgb
XREF    _wwv_pa

		SEGMENT code
; =============================================================================
; Routine send_clk
; Gazinta #1 (IX+6) Byte to write
; Gazoutas: None
; =============================================================================
; This routine takes a byte of data for the outer rings and writes it two 
; bits at a time starting with the LSBs (the shift register is hooked up to 
; the two LSBs of Port C).  Hence routine just writes, shifts two bits, then
; writes again for a total of four writes.  After asserting the data, we wiggle
; bit #3 of port A to clock in the data.
; I speed things up by just having the data in A and rotating it as well as
; having B and C pre-loaded with the PB register values.
; =============================================================================
_send_clk:
		PUSH	IX				; save IX
		LD		IX,0			; 0 so...
		ADD		IX,SP			; IX is now SP

		ld		a,(_wwv_pa)		; get the WWVB state
		or		a,86h			; merge in the rest of the bits for "Clock High"
		ld		b,a				; and save in B

		ld		a,(_wwv_pa)		; get the WWVB state again
		or		a,82h			; merge in the rest of the bits for "Clock Low"
		ld		c,a				; and save in C

		ld		a,(ix+6)		; get the byte to write (will write LSBs first)

		out0	(PC_DR),a		; Assert all 8 bits (only two LSBs latched by HW)
		out0	(PA_DR),b		; Assert clock High
		out0	(PA_DR),c		; Assert clock Low

		rra						; On to next two bits
		rra						; 2nd
		out0	(PC_DR),a		; Assert all 8 bits (only two LSBs latched by HW)
		out0	(PA_DR),b		; Assert clock High
		out0	(PA_DR),c		; Assert clock Low

		rra						; On to next two bits
		rra						; 2nd
		out0	(PC_DR),a		; Assert all 8 bits (only two LSBs latched by HW)
		out0	(PA_DR),b		; Assert clock High
		out0	(PA_DR),c		; Assert clock Low

		rra						; On to the last two bits
		rra						; 2nd
		out0	(PC_DR),a		; Assert all 8 bits (only two LSBs latched by HW)
		out0	(PA_DR),b		; Assert clock High
		out0	(PA_DR),c		; Assert clock Low

		LD		SP,IX			; Omit IF SP is unchanged by routine
		POP		IX				; get original IX back
		RET						; all done!

; =============================================================================
; Routine wrchr
; Gazinta #1 (IX+6) Source address in character memory
; Gazinta #2 (IX+9) Destination address in memory array
; Gazinta #3 (IX+12) # of bits to shift
; Gazoutas: None
; =============================================================================
; This routine copies 8 bytes fom the source character memory to the display
; memory.  As characters can span byte boundaries, the shift parameter says how
; much to shift the byte of data before writing it to memory.  Shifting more 
; than three bits means spanning multiple bytes on writing to display memory.
; =============================================================================
_wrchr:
		PUSH	IX				; save IX
		LD		IX,0			; 0 so...
		ADD		IX,SP			; IX is now SP

		ld		iy,(ix+6)		; IY will be the source address
		ld		hl,(ix+9)		; HL will be the destination address
		ld		c,8				; C will be the byte count for the copy (7->0)

chrlp:	ld		d,(iy)			; get a byte from the character matrix
		ld		e,0				; DE is now 16-bit value to be shifted

		ld		b,(ix+12)		; # of bits to shift
		inc		b				; need it to be +1 so 0 = no shifts
		jr		schrel			; go straight to decrement test
schrlp:	sla		d				; shift LSB to the left (0 into LSbit)
		rl		e				; rotate carry into E
schrel:	djnz	schrlp			; do this B times

; We now have the 5-bit value shifted enough to be or'd into display memory
nchrl:	ld		a,(hl)			; get byte from display memory
		or		a,d				; merge in LSB of shifted data
		ld		(hl),a			; and write it back

		inc		hl				; next byte
		ld		a,(hl)			; get byte from display memory
		or		a,e				; merge in MSB of shifted data
		ld		(hl),a			; and write it back

; OK, we've done one row of the characer, on to the remaining seven.
; we increment the source address IY one byte and the destination
; address HL by four bytes (well three, because it was incremented 
; by one above).  The count to do all eight rows is in C.
		inc		iy				; next byte in the character matrix
		inc		hl				; +2 on destination address
		inc		hl				; +3
		inc		hl				; +4

		dec		c				; decrement row counter
		jr		nz,chrlp		; do all eight rows	
	
		LD		SP,IX			; Omit IF SP is unchanged by routine
		POP		IX				; get original IX back
		RET						; all done!

; =============================================================================
; Routine bfshift
; Gazinta #1 (IX+6) Address of first byte in array
; Gazinta #2 (IX+9) # of bytes to shift through
; Gazoutas: None
; =============================================================================
; This routine does a "right shift" on an array of bytes assuming bits are
; arranged in incrementing bytes with the LSBit on the left.  It isn't quite
; generic in the sense that the LSBit is stuffed with a 1 instead of a 0.
; =============================================================================
_bfshift:
		PUSH	IX				; save IX
		LD		IX,0			; 0 so...
		ADD		IX,SP			; IX is now SP

		ld		hl,(ix+6)		; get the address of the bytes
		ld		b,(ix+9)		; get the # of bytes to shift through

	    scf						; want to start by shifting in a 1
bfslp:	rl		(hl)			; rotate left (Z80 is backwards to my thinking!)
		inc		hl				; next byte
		djnz	bfslp			; do prescibed number of bytes

		LD		SP,IX			; Omit IF SP is unchanged by routine
		POP		IX				; get original IX back
		RET						; all done!

; =============================================================================
; Routine xshiftl
; Gazinta #1 (IX+6) Line #'s to shift (bit0=line1, bit1=line2)
; Gazoutas: None
; =============================================================================
; This routine does a one-pixel left shift from the standby array to the 
; active array.  This does all 16 lines.No params as all are known.
; =============================================================================
_xshiftl:
		PUSH	IX				; save IX
		LD		IX,0			; 0 so...
		ADD		IX,SP			; IX is now SP

		ld		a,(ix+6)		; get the line tags
		push	af				; save it for later

		and		1				; look at bit 0 (for line 1)
		jr		z,xshlc2		; if 0, don't do line 1


xshll1: ld		bc,_dspstga+3	; start with the last byte in the first line of the staging array
		ld		de,_dspmtxa+3	; start with the last byte in the first line of the active array
		call	xshlrt			; go move Phase A

		ld		bc,_dspstgb+3	; start with the last byte in the first line of the staging array
		ld		de,_dspmtxb+3	; start with the last byte in the first line of the active array
		call	xshlrt			; go move Phase B

xshlc2:	pop		af				; get A back
		and		2				; look at bit 1 (for line 2)
		jr		z,xshlex		; if 0, don't do line 2	

xshll2: ld		bc,_dspstga+35	; start with the last byte in the second line of the staging array
		ld		de,_dspmtxa+35	; start with the last byte in the second line of the active array
		call	xshlrt			; go move Phase A

		ld		bc,_dspstgb+35	; start with the last byte in the second line of the staging array
		ld		de,_dspmtxb+35	; start with the last byte in the second line of the active array
		call	xshlrt			; go move Phase B

xshlex:	LD		SP,IX			; Omit IF SP is unchanged by routine
		POP		IX				; get original IX back
		RET						; all done!

xshlrt:	ld		a,8				; wanna do this eight times

xshllp:	ccf						; will clear out staging array as we shift
		ld		hl,bc			; point to the standby array
		rr		(hl)			; Z80s view is opposite of the displays
		dec		hl				; earlier byte (since we are going left)
		rr		(hl)			; byte #2
		dec		hl				; earlier byte (since we are going left)
		rr		(hl)			; byte #3
		dec		hl				; earlier byte (since we are going left)
		rr		(hl)			; byte #4 (last byte in the staging array

		ld		hl,de			; now point to the active array
		rr		(hl)			; rotate into last byte of the active array
		dec		hl				; next byte in active array
		rr		(hl)			; rotate into 3rd byte
		dec		hl				; next byte in active array
		rr		(hl)			; rotate into 2nd byte
		dec		hl				; next byte in active array
		rr		(hl)			; rotate into first byte

		ld		hl,4			; for next line, wanna increment by 7 bytes
		add		hl,bc			; increment staging array addres
		ld		bc,hl			; save it

		ld		hl,4			; for next line, wanna increment by 7 bytes
		add		hl,de			; increment active array addres
		ld		de,hl			; save it

		dec		a				; decrement our line counter
		jr		nz,xshllp		; loop till all 8 lines done 
		ret						; all done

; =============================================================================
; Routine xshiftr
; Gazinta #1 (IX+6) Line #'s to shift (bit0=line1, bit1=line2)
; Gazoutas: None
; =============================================================================
; This routine does a one-pixel right shift from the standby array to the 
; active array.  This does all 16 lines of the display.  No params as all are known.
; =============================================================================
_xshiftr:
		PUSH	IX				; save IX
		LD		IX,0			; 0 so...
		ADD		IX,SP			; IX is now SP

		ld		a,(ix+6)		; get the line tags
		push	af				; save it for later

		and		1				; look at bit 0 (for line 1)
		jr		z,xshrc2		; if 0, don't do line 1

xshrl1:	ld		bc,_dspstga     ; start with the first byte in the first line of the staging array
		ld		de,_dspmtxa     ; start with the first byte in the first line of the active array
		call	xshrrt			; go move Phase A

		ld		bc,_dspstgb     ; start with the first byte in the first line of the staging array
		ld		de,_dspmtxb     ; start with the first byte in the first line of the active array
		call	xshrrt			; go move Phase B

xshrc2:	pop		af				; get A back
		and		2				; look at bit 1 (for line 2)
		jr		z,xshrex		; if 0, don't do line 2	

xshrl2:	ld		bc,_dspstga+32  ; start with the first byte in the second line of the staging array
		ld		de,_dspmtxa+32  ; start with the first byte in the second line of the active array
		call	xshrrt			; go move Phase A

		ld		bc,_dspstgb+32  ; start with the first byte in the second line of the staging array
		ld		de,_dspmtxb+32  ; start with the first byte in the second line of the active array
		call	xshrrt			; go move Phase B

xshrex:	LD		SP,IX			; Omit IF SP is unchanged by routine
		POP		IX				; get original IX back
		RET						; all done!

xshrrt:	ld		a,8				; wanna do this eight times (for each row in the line)

xshrlp:	ccf						; will clear out staging array as we shift
		ld		hl,bc			; point to the standby array
		rl		(hl)			; Z80s view is opposite of the displays
		inc		hl				; earlier byte (since we are going left)
		rl		(hl)			; byte #2
		inc		hl				; earlier byte (since we are going left)
		rl		(hl)			; byte #3
		inc		hl				; earlier byte (since we are going left)
		rl		(hl)			; byte #4 (last byte in the staging array

		ld		hl,de			; now point to the active array
		rl		(hl)			; rotate into last byte of the active array
		inc		hl				; next byte in active array
		rl		(hl)			; rotate into 3rd byte
		inc		hl				; next byte in active array
		rl		(hl)			; rotate into 2nd byte
		inc		hl				; next byte in active array
		rl		(hl)			; rotate into first byte

		ld		hl,4			; for next line, wanna increment by 7 bytes
		add		hl,bc			; increment staging array addres
		ld		bc,hl			; save it

		ld		hl,4			; for next line, wanna increment by 7 bytes
		add		hl,de			; increment active array addres
		ld		de,hl			; save it

		dec		a				; decrement our line counter
		jr		nz,xshrlp		; loop till all 8 lines done 
		ret						; all done

; =============================================================================
; Routine aldir
; Gazinta #1 (IX+6)  Source address
; Gazinta #2 (IX+9)  Destination address
; Gazinta #3 (IX+12) Byte count 
; Gazoutas: None
; =============================================================================
; This routine is a sorta substitute for memcpy targeted toward using the Z80s
; LDIR instruction.
; =============================================================================
_aldir:
		PUSH	IX				; save IX
		LD		IX,0			; 0 so...
		ADD		IX,SP			; IX is now SP

		ld		de,(ix+6)		; get the destination address
		ld		hl,(ix+9)		; get the source address
		ld		bc,(ix+12)		; get the # of bytes to copy
		ldir					; go copy it!
		
		LD		SP,IX			; Omit IF SP is unchanged by routine
		POP		IX				; get original IX back
		RET						; all done!

; =============================================================================
; Routine alddr
; Gazinta #1 (IX+6)  Source address
; Gazinta #2 (IX+9)  Destination address
; Gazinta #3 (IX+12) Byte count 
; Gazoutas: None
; =============================================================================
; This routine is a sorta substitute for memcpy targeted toward using the Z80s
; LDDR instruction.
; =============================================================================
_alddr:
		PUSH	IX				; save IX
		LD		IX,0			; 0 so...
		ADD		IX,SP			; IX is now SP

		ld		de,(ix+6)		; get the destination address
		ld		hl,(ix+9)		; get the source address
		ld		bc,(ix+12)		; get the # of bytes to copy
		lddr					; go copy it!
		
		LD		SP,IX			; Omit IF SP is unchanged by routine
		POP		IX				; get original IX back
		RET						; all done!

; =============================================================================
; Routine rlmem
; Gazinta #1 (IX+6)  Bit to shift in (in bit 0)
; Gazinta #2 (IX+9)  Array starting address
; Gazinta #2 (IX+12) Length of array (byte only!!)
; Gazoutas: None
; =============================================================================
; This routine does a left shift on an array of bytes, starting with the last
; byte in the array and shifting toward the first.  Bits are shifted into the
; LSB first.
; =============================================================================
_rlmem:
		PUSH	IX				; save IX
		LD		IX,0			; 0 so...
		ADD		IX,SP			; IX is now SP

		ld		de,(ix+6)		; get the bit to shift in
		ld		hl,(ix+9)		; get the memory start address
		ld		bc,(ix+12)		; get the # of bytes to shift

		dec		bc				; n bytes are +0 to +n-1 addresses
		add		hl,bc			; HL points to end+1

		ld		b,c				; get the count into B register
		rr		e				; get the bit to shift into C

rlmlp:	rl		(hl)			; rotate left a byte in the array
		dec		hl				; point to earlier byte
		djnz	rlmlp			; do all bytes except first
		rl		(hl)			; now dow the first!

		LD		SP,IX			; Omit IF SP is unchanged by routine
		POP		IX				; get original IX back
		RET						; all done!


; =============================================================================
; Routine
; Gazinta #1 (IX+6) 
; Gazinta #2 (IX+9) 
; Gazoutas: None
; =============================================================================
; This routine 
; =============================================================================
_xxx:
		PUSH	IX				; save IX
		LD		IX,0			; 0 so...
		ADD		IX,SP			; IX is now SP

		LD		SP,IX			; Omit IF SP is unchanged by routine
		POP		IX				; get original IX back
		RET						; all done!


