/*                                         Copyright (c) CMI Productions, 2007
 *****************************************************************************
 *  LedMatrix.c (1.0)
 *
 *  Display rendering functions for the LED matrix
 ****EXTERNAL*****************************************************************
 * ledmatrix_init    : initialize the display "engine"
 * ledmatrix_clear   : All LEDs off
 * ledmatrix_cline   : Erase a text line
 * ledmatrix_pattern : Pretty pattern for LEDs
 * ledmatrix_led_test: LED power-up LED pattern 
 * ledmatrix_putc    : Display a character
 * ledmatrix_puts    : Display a string (sorta)
 * ledmatrix_bset    : set a pixel in the matrix
 * ledmatrix_bclr    : clear a pixel in the matrix
 * ledmatrix_wrtime  : write time to the matrix
 *
 * These are the routines to write to the LED matrix.  Details of the memory
 * structure are in the file "LED Matrix.txt".  Above that document, these
 * routines maintain two arrays of bitmaps, a phase A and phase B.  The
 * display routines alternate between phase A and B every 50ms to get
 * blinking.  The phase parameter is a byte of which only the two lower
 * bits matter.  Phase A is flagged by bit 0 and Phase B by bit 1.
 ****INTERNAL*****************************************************************
 * None
 *****************************************************************************
*/

#include <ez80.h>
#include <string.h>

#include "def.h"
#include "globals.h"
#include "LedMatrix.h"
#include "timer.h"
#include "asmhelp.h"
#include "tables.h"
#include "timer.h"
#include "mystring.h"
#include "transition.h"

/* local global constants and variables (this file only) */

/* internal routines */


/****************************************************************************
* Routne   : ledmatrix_init
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : None (directly)
*
* This routine sets up the LED display process.  First clear bothe LED
* display matrixes to all LEDs off, then fire up the refresh interrupt
* service routines.
****************************************************************************/
void ledmatrix_init( void )
{
   ledmatrix_clear(ACTIVE_PLANE);
   ledmatrix_clear(STANDBY_PLANE);
   ledtimer_init();
   doff = FALSE ;
}

/****************************************************************************
* Routne   : ledmatrix_clear
* Gazintas : plane
* IOs      : None
* Returns  : Nothing
* Globals  : clkmtx : array set to all LEDs off
*            dspmtx : array set to all LEDs off
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine initializes both phases of the displayh matrix to all LEDs 
* off.  0 means an LED off, so this is simply setting both arrays to zeros.
* There are both active and standby planes as well as two phases.  The 
* standby plane has only exists for the matrix display, not the clock
* display.
****************************************************************************/
void ledmatrix_clear( UBYTE plane )
{
   if (plane==ACTIVE_PLANE)
   {
      memset(clkmtxa,0,16) ;
      memset(clkmtxb,0,16) ;

      memset(dspmtxa,0,64) ;
      memset(dspmtxb,0,64) ;
   }
   else if (plane=STANDBY_PLANE)
   {
      memset(dspstga,0,64) ;
      memset(dspstgb,0,64) ;
   }
}

/****************************************************************************
* Routne   : ledmatrix_cline
* Gazintas : plane,line
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array set to all LEDs off
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine clears out a line of the display.  Line 1 is the top line
* top 8 rows) and line 2 is the bottom line (bottom 8 rows).  Line values 
* of 1 and 2 are parsed correctly, other values are ignored.
****************************************************************************/
void ledmatrix_cline( UBYTE plane, UBYTE line )
{
   if (plane==ACTIVE_PLANE)
   {
      if (line==1)
	  {
         memset(dspmtxa,0,32) ;
         memset(dspmtxb,0,32) ;
      }
      else if (line==2)
	  {
         memset(dspmtxa+32,0,32) ;
         memset(dspmtxb+32,0,32) ;
      }

   }
   else if (plane=STANDBY_PLANE)
   {
      if (line==1)
	  {
         memset(dspstga,0,32) ;
         memset(dspstgb,0,32) ;
      }
      else if (line==2)
	  {
         memset(dspstga+32,0,32) ;
         memset(dspstgb+32,0,32) ;
      }
   }
}

/****************************************************************************
* Routne   : ledmatrix_pattern
* Gazintas : plane
* IOs      : None
* Returns  : Nothing
* Globals  : clkmtx : array set to all LEDs alternating
*            dspmtx : array set to all LEDs alternating
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine initializes both matrixes to all LEDs alternating.
****************************************************************************/
void ledmatrix_pattern( UBYTE plane )
{
   UBYTE i ;

   if (plane==ACTIVE_PLANE)
   {
      memset(clkmtxa,0x55,16) ;
      memset(clkmtxb,0xaa,16) ;

      memset(dspmtxa,0x55,64) ;
      memset(dspmtxb,0xaa,64) ;

      for (i=0;i<64;i=i+8)
      {
         memset(dspmtxa+i,0xaa,4) ;
         memset(dspmtxb+i,0x55,4) ;
      }
   }
   else if (plane=STANDBY_PLANE)
   {
      memset(dspstga,0x55,64) ;
      memset(dspstgb,0xaa,64) ;

      for (i=0;i<64;i=i+8)
      {
         memset(dspstga+i,0xaa,4) ;
         memset(dspstgb+i,0x55,4) ;
      }
   }
}

/****************************************************************************
* Routne   : ledmatrix_led_test
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : clkmtx : array set to all LEDs alternating
*            dspmtx : array set to all LEDs alternating
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine shows a pretty pattern at power-up to show that all the LEDs
* actually work
****************************************************************************/
void ledmatrix_led_test ( void )
{
   UBYTE i,j ;

   /* turn off any time displaying */
   showtime(SHOWTIME_OFF) ;

   /* make sure the display is clear first */
   ledmatrix_clear (ACTIVE_PLANE) ;

   /* a sorta useless delay just to get a reference */
   del50ms(1,DEL_NOW) ;

   /* now fill in the two rings of LEDs in a clockwise fasion */
   for (i=0;i<60;i++)
   {
      bfshift(clkmtxa,8) ;
	  bfshift(clkmtxb,8) ;
      bfshift(clkmtxa+8,8) ;
	  bfshift(clkmtxb+8,8) ;
      del50ms(1,DEL_LAST) ;
   } ;

   /* wait for 1 second */
   del50ms(20,DEL_LAST) ;

   /* now fill in the matrix from left to right, one column at a time */
   for (i=0;i<30;i++)
   {
      /* shift in one LED on the top row */
      bfshift(dspmtxa,4) ;

      /* now copy that row to the other rows */
      for (j=0;j<60;j++)
	  {
	     dspmtxa[j+4]=dspmtxa[j] ;
	  } ;

      /* that was Phase A, copy it to phase B as well so it doesn't blink */
      memcpy (dspmtxb,dspmtxa,64) ;

      /* 50ms wait between rows getting filled */ 
      del50ms(1,DEL_LAST) ;
   } ;

   /* now wait for 2 seconds */
   del50ms(40,DEL_LAST) ;

   /* do the blinky pattern for another two seconds */
   ledmatrix_pattern (ACTIVE_PLANE) ;
   del50ms(40,DEL_LAST) ;

   /* OK, enough playing around, clear the screen and return */
   ledmatrix_clear (ACTIVE_PLANE) ;
   ledmatrix_clear (STANDBY_PLANE) ;
}
/****************************************************************************
* Routne   : ledmatrix_putc
* Gazintas : c : The ASCII character to write
*            x : Location to write
*            y : Location to write
*            phase : what phases to write to
*            plane : what plane to write to
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array set to all LEDs off
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine "writes" a character to the LED matrix display.
****************************************************************************/
void ledmatrix_putc( UBYTE x, UBYTE y, UBYTE c, UBYTE phase, UBYTE plane )
{
   UBYTE i        ;
   UBYTE* cptr    ;
   UBYTE* dptra   ;
   UBYTE* dptrb   ;
   UBYTE shift    ;
   UBYTE field[2] ;

   /* basic error checking, just abort if bad data */
   /* Bounds are actually maxx-5 and maxy-8 so character fits */
   if (x>24) return ;
   if (y>8) return ;

   /* do range checking on the character too */
   if ((c<' ')||(c>'~')) return ;
  
   /* point to the first byte in the bitmap for the desired character */
   cptr=&matrix_char_map[c-32][0] ;

   /* decide which matrix array to write on based on the selected plane */
   if (plane==ACTIVE_PLANE)
   {
      dptra=dspmtxa ;
      dptrb=dspmtxb ;
   }
   else if (plane=STANDBY_PLANE)
   {
      dptra=dspstga ;
      dptrb=dspstgb ;
   }

   /* point to the byte in dspmtx for the first write to occur.           */
   /* It is an 8-bit value at an arbitrary address, so I need to know how */
   /* bits to shift it in a 16-bit array to get it written correctly to   */
   /* the display memory.                                                 */
   if (phase&0x01)
   {
      dptra+=(x/8+y*4) ;
      shift=x&0x07 ;
      wrchr(cptr,dptra,shift) ;
   } ;

   if (phase&0x02)
   {
      dptrb+=(x/8+y*4) ;
      shift=x&0x07 ;
      wrchr(cptr,dptrb,shift) ;
   } ;

}

/****************************************************************************
* Routne   : ledmatrix_puts
* Gazintas : str : The string to write (null-terminated)
*            x   : Location to write (TBD)
*            y   : Location to write (TBD)
*            phase : what phases to write to
*            plane : what plane to write to
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : the array to write the text to
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine writes a string to the LED matrix display.  From a practical
* standpoint, the string can't be any longer than 5 characters.  All this 
* routine does is call ledmatrix_putc with characters until a null byte
* is found.  All error checking for X and Y is done by ledmatrix_putc.
****************************************************************************/
void ledmatrix_puts( UBYTE x, UBYTE y, UBYTE *str, UBYTE phase, UBYTE plane )
{
   while (*str!=0)
   {
      ledmatrix_putc(x,y,*str,phase,plane) ;
	  x+=6 ;
	  str++ ;
   }
}

/****************************************************************************
* Routne   : ledmatrix_bset
* Gazintas : x : Pixel location to set (0-29)
*            y : Pixel location to set (0-15)
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array to write to
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine sets the bit at the specified (x,y) location.  Setting a bit
* means turing the LED on.  If coordinates are in error, this routine just
* returns, no error is reported.
****************************************************************************/
void ledmatrix_bset(UBYTE x, UBYTE y, UBYTE phase)
{
   UBYTE idx; /* byte index, 0-63               */
   UBYTE bit; /* bytes bit 0x01, 0x02, ... 0x10 */

   /* basic error checking, just abort if bad data */
   if (x>29) return ;
   if (y>15) return ;

   /* figger out which byte in our matrix and which bit in that byte */
   idx=(15-y)*4+x/8 ;
   bit=oneofeight[x%8] ;
   
   /* to light a pixel, set the bit */
   if (phase&0x01)
   {
      dspmtxa[idx]|=(bit) ;
   } ;
   if (phase&0x02)
   {
      dspmtxb[idx]|=(bit) ;
   } ;
}

/****************************************************************************
* Routne   : ledmatrix_bclr
* Gazintas : x : Pixel location to set (0-29)
*            y : Pixel location to set (0-15)
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array to write to
*            See the file "LED Matrix.txt" for the memory structure
*
* This routine clears the bit at the specified (x,y) location.  Clearing a bit
* means turing the LED off.  If coordinates are in error, this routine just
* returns, no error is reported.
****************************************************************************/
void ledmatrix_bclr(UBYTE x, UBYTE y, UBYTE phase)
{
   UBYTE idx; /* byte index, 0-63               */
   UBYTE bit; /* bytes bit 0x01, 0x02, ... 0x10 */

   /* basic error checking, just abort if bad data */
   if (x>29) return ;
   if (y>15) return ;

   /* figger out which byte in our matrix and which bit in that byte */
   idx=(15-y)*4+x/8 ;
   bit=oneofeight[x%8] ;
   
   /* to clear a pixel, clear the bit */
   if (phase&0x01)
   {
      dspmtxa[idx]&=(~bit) ;
   } ;
   if (phase&0x02)
   {
      dspmtxb[idx]&=(~bit) ;
   } ;

}

/****************************************************************************
* Routne   : ledmatrix_wrtime
* Gazintas : Mode: If setting the time/date, which one is being set
*            trtype: transition type
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array to write to
*            See the file "LED Matrix.txt" for the memory structure
*            hrs, mins, secs, ssecs : the time to write 
*
* This routine writes out the time and date to the LED matrix.  If we are in
* time-setting mode (dhold=TRUE), then we use mode to decide which parameter
* to blink.  If not setting the time, then trtype is used to decide what
* kind of transition to use for the new time.
****************************************************************************/
void ledmatrix_wrtime( UBYTE mode, UBYTE trtype )
{
   UBYTE str[6] ; /* the place to put the date/time */
   UBYTE tmp    ; /* for some hours math            */
   UBYTE x=0    ; /* Starting location for time     */

   if (mode!=SHOWTIME_OFF)
   {
      ledmatrix_clear(STANDBY_PLANE) ;
      if (1) /* This will be for the update flag later) */
      {
         tmp=hrs ;              /* start with 00-23 hours  */
         if (!thold)            /* if not setting time...  */
		 {
   	        if (tmp>12) tmp-=12 ;  /* do PM correctly         */
	        if (tmp==0) tmp=12  ;  /* do midnight correctly   */
		 } ;
         itoa2(tmp,str)      ;  /* result in 1st two bytes */
         str[2]=':'          ;  /* the semicolon           */
         itoa2(mins,str+3)   ;  /* write the minites       */
         str[5]=0            ;  /* terminate the string    */

         /* now that we have the time, if there is a leading 0, omit it and center the time */
         tmp=0 ;
	     if (str[0]=='0')
         {
            tmp=1 ;
            x=3   ;
		 }
		 
		 ledmatrix_puts(x,0,str+tmp,PHASEA,STANDBY_PLANE) ; /* write it to the display Phase A */

		 if (mode==SHOWTIME_SETHRS) /* if setting hours, write colon and minutes to Phase B */
		 {
		    str[0]=' ' ; /* "erase" hours so it won't be in phase B */
			str[1]=' ' ;
		 } else if (mode==SHOWTIME_SETMINS) /* if setting minutes, write colon and hours to Phase B */
		 {
		    str[3]=' ' ; /* "erase" minutes so it won't be in phase B */
			str[4]=' ' ;
         }
	     ledmatrix_puts(x,0,str+tmp,PHASEB,STANDBY_PLANE) ; /* write it to the display Phase A */

         /* now on to the date */

         itoa2(month,str)    ;  /* wrute the month         */
         str[2]='/'          ;  /* the semicolon           */
         itoa2(day,str+3)    ;  /* write the day           */
         str[5]=0            ;  /* terminate the string    */
         ledmatrix_puts(0,8,str,PHASEA,STANDBY_PLANE) ; /* write it to the display Phase A */

		 if (mode==SHOWTIME_SETMONTH) /* if setting hours, write colon and minutes to Phase B */
		 {
		    str[0]=' ' ; /* "erase" hours so it won't be in phase B */
			str[1]=' ' ;
		 } else if (mode==SHOWTIME_SETDAY) /* if setting minutes, write colon and hours to Phase B */
		 {
		    str[3]=' ' ; /* "erase" minutes so it won't be in phase B */
			str[4]=' ' ;
         }
	     ledmatrix_puts(0,8,str,PHASEB,STANDBY_PLANE) ; /* write it to the display Phase A */

         /* now that we've drawn everything on the standby plane, copy it to the active plane */
		 /* for normal mode, just to a CUT transition, for fancy mode, pick a mode based on   */
		 /* the current minute.                                                               */
		 tmp=(trtype==TRT_NORMAL)?0:mins%15 ;
         /* use the transition table to pick the transition type */
         transition(trtbl[tmp][0],trtbl[tmp][1],TR_LINE12,FALSE) ;   

      } 
   }
}

/****************************************************************************
* Routne   : ledmatrix_wryear
* Gazintas : Mode: The display mode.
* IOs      : None
* Returns  : Nothing
* Globals  : dspmtx : array to write to
*            See the file "LED Matrix.txt" for the memory structure
*            year : the year to write 
*
* This routine writes out the year to the LED matrix.  This is only used
* while setting the year, so unless display mode is off, this will always
* blink the last two digits.
****************************************************************************/
void ledmatrix_wryear( UBYTE mode )
{
   UBYTE str[6] ; /* the place to put the date/time */
   UBYTE tmp    ; /* for some hours math            */

   if (mode!=SHOWTIME_OFF)
   {
      ledmatrix_clear(ACTIVE_PLANE) ;
      if (1) /* This will be for the update flag later) */
      {
         str[0]='2'          ;  /* '2' of 20xx             */
         str[1]='0'          ;  /* '0' of 20xx             */
         itoa2(year,str+2)   ;  /* fill in year            */
         str[4]=0            ;  /* terminate the string    */
         ledmatrix_puts(3,0,str,PHASEA,ACTIVE_PLANE) ; /* write it to the display Phase A (centered) */

		 if (mode==SHOWTIME_SETYEAR) /* if setting year, make it blink to Phase B */
		 {
		    str[2]=' ' ; /* "erase" hours so it won't be in phase B */
			str[3]=' ' ;
		 }
	     ledmatrix_puts(3,0,str,PHASEB,ACTIVE_PLANE) ; /* write it to the display Phase A */

      } 
   }
}

