/*                                         Copyright (c) CMI Productions, 2007
 *****************************************************************************
 *  wwvb.c (1.0)
 *
 *  WWVB reception functions
 ****EXTERNAL*****************************************************************
 * wwvb_init: Initialize the WWVB radio (at power-up)
 * wwvb_go:   Start the WWVB state machine 
 * wwvb_stat: Get the state machine status
 * wwvbsm:    WWVB state machine (does the receiving of the signal) 
 * wwvlsc:    Check for leap seconds.
 * 
 ****INTERNAL*****************************************************************
 * wwvb_init()  ; initialize the variables for WWVB reception.
 * wwvb_sync()  ; wait for two sync (Px) symbols.
 * wwvb_gdata() ; get 60 bits of WWVB data.
 * wwvb_proc1() ; convert WWVB data to UTC.
 * wwvb_proc2() ; convert UTC to local time.
 * wwvb_stime() ; Actually set the clocks time!
 *****************************************************************************
 * To Do Still:
 *       Right now this code does the basic WWVB decode and time setting
 *      to UTC (not UT1).  There are still a lot of things to do still
 *      1! Check all the boundary conditions for time setting--there are a lot!
 *         Leap seconds, DST transitions, other dates...
 *      2! Set up the code to be called once a day, not just at power-up
 *      3! User-settable time zone and DST observation
 *      4) Error checking (by getting the time twice)
 *      5) Calibrating the time (offset checking afer 24 hours) (may not implement)
 *      6) Noise filtering of the WWVB signal
 *      7! Confirm leap-seconds move to midnight UTC, not midnight local
 *
 * Written but not tested yet:
 *   1) 5-minute timeout on getting time
 *   2) New symbol-getting routine that responds every second no matter what
 */

#include <ez80.h>
#include <string.h>

#include "def.h"
#include "gpio.h"
#include "globals.h"
#include "tables.h"
#include "wwvb.h"
#include "asmhelp.h"
#include "menus.h"

/* Local Defines */
#define SYM_NONE 0
#define SYM_1    1
#define SYM_0    2
#define SYM_P    3
#define SYM_BAD  4

/* This is the debug flag for checking boundary conditions of the WWVB   */
/* decoder.  If in debug mode, I preload a sequence as if it came from   */
/* the WWVB receier in the wwvb_prep routine, then jump to the wwv_proc1 */
/* reoutine to decode it straight away.  See the table below for the     */
/* various boundary conditions being tested.                             */
#define WWV_DEBUG FALSE

/* Local globals */
UBYTE wwvbst=WWVB_IDLE ; /* The WWVB state machine state */
UWORD wwv_timeout=0    ; /* WWVB receive timeout counter */
UBYTE wwvbits[8]       ; /* The array of bits from WWVB  */
UBYTE bit_counter      ; /* WWVB bit counter             */
UBYTE ww_mins          ; /* Minutes from WWVB data       */
BYTE  ww_hours         ; /* Hours from WWVB data (can be negative) */
UWORD ww_days          ; /* Day in year from WWVB data   */
UBYTE ww_year          ; /* Year from WWVB data          */
UBYTE ww_day           ; /* day of month                 */
UBYTE ww_month         ; /* month in year                */

UBYTE gsym             ; /* Global version of symbol just parsed by WWVB Rx     */
UBYTE gcnt             ; /* Global version of time count just parsed by WWVB Rx */

/* table to convert 50ms ticks to WWVB symbols (see symbol in wwvb.c) */
const UBYTE sym_table[21] = {
   SYM_BAD, /* 0,  0ms       */
   SYM_BAD, /* 1,  50ms      */
   SYM_0,   /* 2,  100ms     */
   SYM_0,   /* 3,  150ms     */
   SYM_0,   /* 4,  200ms "0" */
   SYM_0,   /* 5,  250ms     */
   SYM_BAD, /* 6,  300ms     */
   SYM_BAD, /* 7,  350ms     */
   SYM_1,   /* 8,  400ms     */
   SYM_1,   /* 9,  450ms     */
   SYM_1,   /* 10, 500ms "1" */
   SYM_1,   /* 11, 550ms     */
   SYM_BAD, /* 12, 600ms     */
   SYM_BAD, /* 13. 650ms     */
   SYM_P,   /* 14, 700ms     */
   SYM_P,   /* 15, 750ms     */
   SYM_P,   /* 16, 800ms "P" */
   SYM_P,   /* 17, 850ms     */
   SYM_BAD, /* 18, 900ms     */
   SYM_BAD, /* 19, 950ms     */
   SYM_BAD};/* 20,1000ms     */

const UBYTE marker_table[60] = {
   1,0,0,0,0,   /* 0- 4: Position marker table.                  */
   0,0,0,0,1,   /* 5- 9: 1 for when we expect a position marker, */
   0,0,0,0,0,   /*10-14: 0 for when we expect data               */
   0,0,0,0,1,   /*15-19: So we can do rudimentary error checking */
   0,0,0,0,0,   /*20-24  */
   0,0,0,0,1,   /*25-29  */
   0,0,0,0,0,   /*30-34  */
   0,0,0,0,1,   /*35-39  */
   0,0,0,0,0,   /*40-44  */
   0,0,0,0,1,   /*45-49  */
   0,0,0,0,0,   /*50-54  */
   0,0,0,0,1} ; /*55-59  */

#if WWV_DEBUG

/* which of the sequences to test */
#define WWV_DEBUG_SEQ 16

const UBYTE wwv_primer[17][8]= {
   {0x03,0x18,0x85,0x22,0x84,0x45,0x02,0x16}, /* #0 11:34pm UTC 9/7/08  DST (my first live debugging)        */
   {0x03,0x18,0x86,0x02,0xA4,0xA0,0x02,0x16}, /* #1 11:34pm UTC 2/28/08 DST (checking leap year)             */
   {0x03,0x18,0x86,0x03,0x00,0xA0,0x02,0x16}, /* #2 11:34pm UTC 2/29/08 DST (checking leap year)             */
   {0x03,0x18,0x86,0x03,0x04,0xA0,0x02,0x16}, /* #3 11:34pm UTC 3/1/08  DST (checking leap year)             */
   {0x03,0x18,0x86,0x02,0xA4,0xA0,0x01,0xC6}, /* #4 11:34pm UTC 2/28/07 DST (checking non leap year)         */
   {0x03,0x18,0x86,0x03,0x00,0xA0,0x01,0xC6}, /* #5 11:34pm UTC 3/1/07  DST (checking non leap year)         */
   {0x03,0x18,0x86,0x14,0xA4,0xA0,0x02,0x16}, /* #6 11:34pm UTC 7/17/08 DST (day 199, LY)  (only for 6808)   */
   {0x03,0x18,0x86,0x20,0x00,0xA0,0x02,0x16}, /* #7 11:34pm UTC 7/18/08 DST (day 200, LY)  (only for 6808)   */
   {0x03,0x18,0x86,0x14,0xA4,0xA0,0x01,0xC6}, /* #8 11:34pm UTC 7/18/07 DST (day 199, NLY) (only for 6808)   */
   {0x03,0x18,0x86,0x20,0x00,0xA0,0x01,0xC6}, /* #9 11:34pm UTC 7/19/07 DST (day 200, NLY) (only for 6808)   */
   /* Check out DST transitions (already checked out DST/not)                                                */
   /* Run these to see.                                                                                      */
   {0x05,0x40,0x10,0x04,0x88,0xA0,0x02,0x14}, /* #10  8:59am UTC 4/1/08 ST-DST transition (1:59am MST)       */
   {0x05,0x40,0x40,0x04,0x88,0xA0,0x02,0x14}, /* #11 10:59am UTC 4/1/08 After ST-DST transition (4:59am MDT) */
   {0x05,0x40,0x0e,0x23,0x94,0xA0,0x02,0x12}, /* #12  7:59am UTC 10/1/08 DST-ST transition (1:59am MDT)      */
   {0x05,0x40,0x12,0x23,0x94,0xA0,0x02,0x12}, /* #13  9:59am UTC 10/1/08 After DST-ST transition (2:59am MST)*/
   /* Leap seconds tests                                                                                     */
   {0x05,0x40,0x86,0x33,0x18,0xA0,0x02,0x10}, /* #14 11:59pm UTC 12/31/08 No Leap Second (4:59pm MST)        */
   {0x05,0x40,0x86,0x33,0x18,0xA0,0x02,0x18}, /* #15 11:59pm UTC 12/31/08 With Leap Second (4:59pm MST)      */
   /* Test to make sure I get the time at local midnight                                                     */
   /* Also stress test of backing out time for local offset!                                                 */
   {0x05,0x40,0x0c,0x04,0x88,0xA0,0x02,0x10}};/* #16  6:59am UTC 4/1/08 (11:59pm MST)                        */


#endif

/* Local Routines */
void  wwvb_prep ( void ) ; 
void  wwvb_sync ( void ) ; 
void  wwvb_gdata( void ) ; 
void  wwvb_proc1( void ) ; 
void  wwvb_proc2( void ) ; 
void  wwvb_stime( void ) ; 
UBYTE symbol    ( void ) ;

/****************************************************************************
* Routne   : wwvb_init
* Gazintas : None
* IOs      : None
* Returns  : None
* Globals  : wwvb_led, wwv_pa, observe_DST, timezone, dstflag
*
* This routine initializes some variables that need it at startup.
****************************************************************************/
void wwvb_init (void)
{
   wwvb_led=0          ; /* the state for the LED showing the WWVB signal        */
   wwv_pa=0x20         ; /* Bit 5 of the radio enable bit (default disabled)     */
   dstflag=0           ; /* DST live transition flag.                            */
   nsecs=60            ; /* default # of seconds in a minute (for leap seconds!) */
   wwv_timeout=0       ; /* reset WWVB receive timeout counter                   */
   memset(wwvbits,0,8) ; /* clear out our WWVB array                             */
}

/****************************************************************************
* Routne   : wwvb_go
* Gazintas : None
* IOs      : None
* Returns  : None
* Globals  : wwvbst
*
* This is the routine to fire off the WWVB state machine.  It simply kicks
* the state machine variable out of the idle state.  Note that this can also
* restart the state machine if it gets hung for some reason.
****************************************************************************/
void wwvb_go (void)
{
  wwvbst=WWVB_PREP ;
}

/****************************************************************************
* Routne   : wwvb_stat
* Gazintas : None
* IOs      : None
* Returns  : WWVB State machine state
* Globals  : wwvbst
*
* This routine just returns the state of the WWVB state machine.  Note that 
* for sync reasons, gsym (global sybol value) gets reset on read.
****************************************************************************/
UBYTE wwvb_stat (UBYTE var)
{
   UBYTE tmp=0 ;
   
   if (var==WWVB_RSTATE) tmp=wwvbst            ;
   if (var==WWVB_RBIT)   tmp=bit_counter       ;
   if (var==WWVB_SYM)  { tmp=gsym ; gsym=0 ; } ;
   if (var==WWVB_CNT)    tmp=gcnt              ;

   return(tmp) ;
}

/****************************************************************************
* Routne   : wwvb_last
* Gazintas : None
* IOs      : month,day,hour,minute: Time of last acquisition
* Returns  : Nothing
* Globals  : 
*
* This routine returns date and time of the last acquisition.
****************************************************************************/
void wwvb_last (UBYTE* month, UBYTE* day, UBYTE* hour, UBYTE* minute )
{
   *month  = ww_month  ;
   *day    = ww_day    ;
   *hour   = ww_hours  ; /* some casting here! */
   *minute = ww_mins   ;
}

/****************************************************************************
* Routne   : wwvbsm
* Gazintas : None
* IOs      : None
* Returns  : None
* Globals  : (the time)
*
* This is the main state machine for parsing the WWVB signal and converting
* it to local time.  It gets called every 50ms.  If it is not getting the
* time it returns immedidately.  For getting the time, it sequences through
* a series of states to sync, acquire, process, and set the time.
****************************************************************************/
void wwvbsm (void)
{

   /* if we're idle, return fast and don't waste CPU cycles */
   if (wwvbst==WWVB_IDLE) return ;

   /* save the WWVB input state for blinking the status */
   wwvb_led=gpio_get_wwvb() ;

   /* if we are actively receiving WWVB, blink the noon tick mark */
   clkmtxa[8]=(clkmtxa[8]&0xfe)|wwvb_led ;
   clkmtxb[8]=(clkmtxb[8]&0xfe)|wwvb_led ;

   /* our state machine! */
   switch (wwvbst)
   {
      case WWVB_PREP  : wwvb_prep()      ; break ;
      case WWVB_SYNC1 : wwvb_sync()      ; break ;
      case WWVB_SYNC2 : wwvb_sync()      ; break ; /* yes, twice! */
      case WWVB_GDATA : wwvb_gdata()     ; break ;
      case WWVB_PROC1 : wwvb_proc1()     ; break ;
      case WWVB_PROC2 : wwvb_proc2()     ; break ;
      case WWVB_STIME : wwvb_stime()     ; break ;
	  default         : wwvbst=WWVB_IDLE ; break ; /* if bad state, stop! */
   }

   /* OK, now see if we are taking too long.  Increment a counter for each */
   /* call to this routine.  5 minutes at 50ms is 6000 calls.  If after    */
   /* 6000 calls we haven't successfully received the time, give up.       */
   /* This counter only counts when actively getting time and is reset in  */
   /* the wwvb_prep routine.                                               */
   /* By going straight to idle, the original time is left alone.          */
   if ((wwv_timeout++)==6000) wwvbst=WWVB_IDLE ;

}

/****************************************************************************
* Routne   : wwvb_init
* Gazintas : None
* IOs      : None
* Returns  : None
* Globals  : wwvbits, bit_counter
*
* This routine does the one-time initializing of the variables for getting
* the time from WWVB. We turn on the radio, clear the bits array and reset
* the bit counter and bit timer.
****************************************************************************/
void wwvb_prep( void )   
{
   gpio_ena_wwvb(TRUE) ; /* turn on the WWVB receiver */
   memset(wwvbits,0,8) ; /* clear out our array       */
   bit_counter=1       ; /* clear our bit counter     */
   wwv_timeout=0       ; /* reset WWVB timeout counter*/
   wwvbst=WWVB_SYNC1   ; /* and on to the next state  */

   /* See define in this files header for full explanation */
   /* If debugging,copy a line and jump to the parser      */
#if WWV_DEBUG
   memcpy (wwvbits,wwv_primer[WWV_DEBUG_SEQ],8) ;
   wwvbst=WWVB_PROC1 ;
#endif

}

/****************************************************************************
* Routne   : wwv_sync
* Gazintas : None
* IOs      : None
* Returns  : None
* Globals  : 
*
* This routine hangs around for two consecutive sync (P) symbols to know 
* where to start getting data.  This actually occupies two states of the 
* WWVB state machine state counter.
****************************************************************************/
void wwvb_sync( void )   
{
   UBYTE sym ; /* the symbol retrieved */

   /* If we have a marker bit, increment the state, otherwise reset to */
   /* WWVB_SYNC1.  That way, the only way to get out of this state is  */
   /* to get two marker bits in a row.                                 */
   /* Of course, only act if there *is* a symbol of some kind.         */
   sym=symbol() ;
 
   if (sym!=SYM_NONE)
   {
      if (sym==SYM_P)
      {
         wwvbst++ ;
      }
      else
      {
         wwvbst=WWVB_SYNC1 ;
      }
   }
}

/****************************************************************************
* Routne   : wwv_gdata
* Gazintas : None
* IOs      : None
* Returns  : None
* Globals  : wwvbits
*
* This routine gets the 60 bits of data from the WWVB transmission once
* syncronization has been established.  As the bits come in they are shifted
* into the wwvbits array.  As bits come in, I check to see that marker 
* bits come in as expected (and not) and if the sequence gets out of whack
* start over.
****************************************************************************/
void wwvb_gdata( void )  
{
   UBYTE sym ; /* the symbol retrieved */

   sym=symbol() ;
 
   switch(sym)
   {
      /* if no symbol on this call, do nuthin */
      case SYM_NONE : break ;
	  /* Treatment for 1's and 0's are the same */
	  case SYM_0    :
	  case SYM_1    : /* make sure we are expecting a 1/0 bit */
	                  if(marker_table[bit_counter]==0) 
	                  {
					     /* if so, shift it in and increment bit counter */
					     rlmem(sym,wwvbits,8) ;
                         bit_counter++ ;
                      }
					  else
					  {
					     /* if not, start over */
					     wwvbst=WWVB_PREP ;
                      } ;
					  break ;
	  case SYM_P    : /* make sure we are expecting a marker bit */
	                  if(marker_table[bit_counter]==1)
	                  {
					     /* if so, shift in a 0 as a place holder and increment bit counter */
					     rlmem(0,wwvbits,8) ;
                         bit_counter++ ;
                      }
					  else
					  {
					     /* if not, start over */
					     wwvbst=WWVB_PREP ;
                      } ;
					  break ;
	  default       : /* anything but 1/0/P bits is an error, start over */
	                  wwvbst=WWVB_PREP ;
					  break ;
   } ;
   /* Debug code */
   if (wwvbst==WWVB_PREP)
   {
     bit_counter=0 ;
   }


   /* if we got to 60 without errors, on to next state */
   if (bit_counter==60) wwvbst=WWVB_PROC1 ;   
}

/****************************************************************************
* Routne   : wwv_proc1
* Gazintas : None
* IOs      : None
* Returns  : None
* Globals  : wwvbits, wwv time (in UTC)
*
* This is the routine that takes the WWVB data and converts it
* it to (mostly) local format.  I leave it in day-of-year format at
* at least for now until after I do all the conversions to actual 
* local time.  I parse the time and write it to the
* holding bytes before writing it to the official time.
* Since WWVB time is BCD and I'm binary, there is quite a bit
* of converting going on!  Note that I have a 64-bit buffer, but
* only 59 of the 60 bits get shifted in. I don't bother shifting in
* the initial Pr bit.  This means that my array is offset somewhat
* from the WWVB table.  The Minutes 40-bit is in byte 0, bit 2.
*  10s-Minutes = byte0[2:0]          (3 bits)
*   1s-Minutes = byte1[6:3]          (4 bits)
*  10s-Hours   = byte2[7:6]          (2 bits)
*   1s-Hours   = byte2[4:1]          (4 bits)
* 100s-DOY     = byte3[5:4]          (2 bits)
*  10s-DOY     = byte3[2:0]byte4[7]  (4 bits)
*   1s-DOY     = byte4[5:2]          (4 bits)
*  10s-Year    = byte6[6:3]          (4 bits)
*   1s-Year    = byte6[1:0]byte7[7:6 (4 bits)
* The various marker bits (leap year, leap second, and DST) don't
* need parsing.  This just decodes the time as sent by WWVB,
* the next state converts it to localt time.
****************************************************************************/
void wwvb_proc1( void )  
{
   /* run the symbol routine to keep tming accurate even though I have */
   /* no interest in the result (it had better be SYM_NONE)            */
   symbol() ;

   /* some pretty heavy duty bit twiddling */
   ww_mins  = ((wwvbits[0]   )&0x07)*10  + ((wwvbits[1]>>3)&0x0f)    ;
   ww_hours = ((wwvbits[2]>>6)&0x03)*10  + ((wwvbits[2]>>1)&0x0f)    ;
   ww_days  = ((wwvbits[3]>>4)&0x03)*100 + ((wwvbits[3]   )&0x07)*20 +
              ((wwvbits[4]>>7)&0x01)*10  + ((wwvbits[4]>>2)&0x0f)    ;
   ww_year  = ((wwvbits[6]>>3)&0x07)*10  + ((wwvbits[6]   )&0x03)*4  +
              ((wwvbits[7]>>6)&0x03)                                 ;

   wwvbst=WWVB_PROC2 ;   
}

/****************************************************************************
* Routne   : wwv_proc2
* Gazintas : None
* IOs      : None
* Returns  : None
* Globals  : ww_times
*
* This routine takes the UTC time decoded in wwb_proc1 and converts it to
* actual local time, including timezone offset, DST, and the +1 minute adder
* because the time sent is a minute behind.
* I also convert day-of-year to day/month.
* See note on day fixing, I do resort to using the year to determine leap
* years and not the leap year flag in the WWVB stream.
****************************************************************************/
void wwvb_proc2( void )  
{
   UWORD loc_days ; /* local day counter */
   UWORD dpm      ; /* days per month    */

   /* run the symbol routine to keep tming accurate even though I have */
   /* no interest in the result (it had better be SYM_NONE)            */
   symbol() ;

   /* increment the minutes by 1 and (if needed) hours too.            */
   /* Worry about hours "overflow" a bit later, after other adjustments*/
   ww_mins++ ;
   if (ww_mins>=60)
   {
      ww_mins=0 ;
	  ww_hours++ ;
   } ;

   /* Now for the messy DST stuff.  Two bits in byte 7 say what to do. */
   /* Bits [2:1] have the following meanings                           */
   /* 00: Standard time all day today                                  */
   /* 11: Daylight Saving time all day today                           */
   /* 10: Transition to DST today                                      */
   /* 01: Transition to ST                                             */
   /* Only do any of this if DST observation is turned on.             */

   /* The dstflag is to trigger a DST change, so default is nothing.   */
   dstflag=NOTHING ;

   if (get_dso()) /* see if we are observing DST */
   {
      switch ((wwvbits[7]>>1)&0x03) /* turn two bits into an integer     */
	  {
	     case 0 : /* Standard time all day -- do nothing                 */
		          break ;
         case 1 : /* Here if we are transitioning to Standard time today */
		          /* (fall back).  If the standard time is less than 1am */
				  /* assume DST, but set flag to go to ST at 2am.        */
				  if ((ww_hours+get_tzo())<1)
				  {
				     ww_hours++ ;
					 dstflag=TO_ST ;
                  } ;
		          break ;
         case 2 : /* Here if we are transitioning to DST today (spring   */
		          /* forward).  If the standard time is after 2am        */
				  /* assume DST, otherwise set flag to go to DST at 2am. */
				  if ((ww_hours+get_tzo())>=2)
				  {
				     ww_hours++ ;
                  }
                  else
				  {
					 dstflag=TO_DST ;
                  } ;
				  break ;
         case 3 : /* DST all day, just increment 1 hour                  */
		          ww_hours++ ;
		          break ;
      }
   }

   /* With all the DST stuff taken care of, now take care of the offset */
   /* to local time.                                                    */
   ww_hours+=get_tzo() ;

   /* That takes care of hours offsets.  We haven't bothered with the   */
   /* impact to the days yet, letting all the hours adjustments take    */
   /* place.  Hours can now be less than 0 or greater than 23, so start */
   /* fixing that.                                                      */

   /* if we rolled back a day */
   if (ww_hours<0)
   {
      ww_hours+=24 ;
	  ww_days--    ;
	  if (ww_days==0)
	  {
		 if (ww_year!=0)   ww_year--     ; else ww_year=99   ;
         /* avoid the leap-year mess by directly assigning 12/31! */
		 ww_day  =31 ;
		 ww_month=21 ;
		 ww_days =0  ; /* ww_days=0 means we have a date set */
      }
   }

   /* if we rolled forward a day */
   if (ww_hours>23)
   {
      ww_hours-=24 ;
	  ww_days++    ;
	  /* first figger out how many days this year to determine max */
      if (wwvbits[7]&0x10) loc_days = 366 ; else loc_days= 365 ;
	  if (ww_days>=loc_days) ww_days=1 ;
   }

   /* OK that is the official local time/date except that we have */
   /* day-of-year instead of date/month.  Convert that now.       */
   /* Of note, in WWVB-ese, Jan 1st is day 1, not day 0, so I     */
   /* subtract one first before counting off months.  Also, a     */
   /* ww_days value of 0 means I already have 12/31.              */
   if (ww_days!=0)
   {
      loc_days=ww_days-1 ;
      ww_month=0 ;
      dpm=31 ;

      while (loc_days>=dpm)
      {
         loc_days-=dpm ;
         ww_month++ ;
         dpm=days_per_month[ww_month] ;
         if ((ww_month==1)&&(wwvbits[7]&0x10)) dpm=29 ;
      }

      /* Days should be correct range,but need to adjust months from */
      /* 0-11 to 1-12.  Day of month is right, just need to make it  */
      /* global for the next state.                                  */
      ww_month++ ;   
      ww_day=(UBYTE)loc_days+1 ;
   }

   wwvbst=WWVB_STIME ;   

}

/****************************************************************************
* Routne   : wwv_sync
* Gazintas : None
* IOs      : None
* Returns  : None
* Globals  : The real time
*
* This routine gets called after we've processed the time and are now
* waiting for the leading edge of the Px marker to actually set the local
* time to the decoded time.  wwv_led is the state of the transmission, so
* I use that instead of calling the GPIO routine
****************************************************************************/
void wwvb_stime( void )  
{
   if (wwvb_led) /* wait for the transmission to start */
   {
      ssecs =0        ; /* Sub-seconds to 0 */
      secs  =0        ; /* Seconds to 0     */
      mins  =ww_mins  ; /* Copy Minutes     */
      hrs   =ww_hours ; /* Copy Hours       */
      day   =ww_day   ; /* Copy Day         */
      month =ww_month ; /* Copy Month       */
      year  =ww_year  ; /* Copy Year        */
	  newmin=TRUE     ; /* Update the clock!*/

      /* once the time is set, radio off and done!       */
      gpio_ena_wwvb(FALSE) ;
      wwvbst=WWVB_IDLE ;   
  }
}

/****************************************************************************
* Routne   : symbol
* Gazintas : None
* IOs      : None
* Returns  : WWVB Symbol (if we got one)
* Globals  : 
*
* This routine gets called every 50ms we are looking for symbols.  Most of
* the time this routine returns SYM_NONE, meaning just wait.  Something 
* gets returned on any rising edge of the incoming signal based on timing.
* Also, if there is no transition for at least one second, this routine 
* returns SYM_BAD just to keep things from appearing locked.  Symbols that
* get returned are based on the following timings:
*    0 bit = 200ms = 3-5 high samples   = returns SYM_0
*    1 bit = 500ms = 9-11 high samples  = returns SYM_1
*    P bit = 800ms = 15-17 high samples = returns SYM_P
*    anything else is gunk              = returns SYM_BAD
*
****************************************************************************/
UBYTE symbol(void)
{
   static UBYTE bit_timer=0     ; /* WWVB bit counter             */
   static UBYTE last_state=0xff ; /* last state (init to invalid) */
   UBYTE sym=SYM_NONE           ; /* the symbol to return         */
   UBYTE temp=0 ;

   /* if no change in state, just increment the bit timer     */
   /* if it goes longer than 1 sec, return an error and reset */
   if (wwvb_led==last_state)
   {
      bit_timer++  ;

	  if (bit_timer>20)
	  {
	     bit_timer=0 ;
		 sym=SYM_BAD ;
	  }
   }
   else /* we had a change in state! */
   {
      if (!wwvb_led) /* if change was to low, process symbol! */
      {
	     /* it isn't possible to get a count >20, so no need for */
		 /* bounds checking.                                     */
         sym=sym_table[bit_timer] ;
      }
      /* Don't care about a low-to-high transitions.    */
	  /* No matter what state change, reset the counter */
      temp=bit_timer ;
      bit_timer=0 ;
   }

   /* now save the current state as last_state so we can find transitions */
   last_state=wwvb_led ;

   /* this is for debug, going to leave it for a bit */
   if (sym!=SYM_NONE)
   {
	  gsym=sym                     ;
	  gcnt=temp                    ;
//      dspmtxa[32]=dspmtxb[32]=temp ;

   }
   return(sym) ;
}

/****************************************************************************
* Routne   : wwvlsc
* Gazintas : None
* IOs      : None
* Returns  : WWVB Symbol (if we got one)
* Globals  : The time
*            WWVB bit stream
*
* This routine figgers out if we need to implement a leap second
* or not.  Leap seconds extend the last second of the day in a
* month which turns out hard to do efficiently inside the time
* incrementing rouinte, so instead I do it as a seperate routine
* that runs less often.  I check in order of things likely to
* disqualify the event as a leap second to spend the least time
* in this routine as possible.  The output of this routine is
* the number to compare seconds to in order to determine if we
* should increment the minutes.  This is normally 60 except for
* when there is a leap second, in which case it is 61.
* A leap second happens when:
*  - The leap-second flag is set (wwvbits[7] bit 3)
*  - It is the last day of the month
*  - It is 11:59pm UTC
*
****************************************************************************/
void wwvlsc ( void )
{
   UBYTE dpm ; /* days in the month */
   BYTE  gmt ; /* GMT hours         */

   nsecs=60 ; /* The default # of seconds */

   /* if no leap second flag, exit */
   if ((wwvbits[7]&0x08)==0) return ;

   /* if not the last day in the month, exit */
   dpm=days_per_month[month-1] ;
   if ((month==2)&&(wwvbits[7]&0x10)) dpm=29 ;
   if (day!=dpm) return ;

   /* if not 11pm GMT, exit */
   gmt = (BYTE)hrs-get_tzo()  ; /* baseline GMT by subtracting local offset  */
   if (wwvbits[7]&0x02) gmt-- ; /* if DST or day changing to ST, apply offst */
   if (gmt<0) gmt+=24         ; /* if underflow, fix it                      */
   if (gmt!=23) return ;      ; /* if all that and not 11pm return           */

   /* only if 59 minutes is leap-second possible */
   if (mins!=59) return ;

   /* OK! Flag set, last day of the month, and it's 11:59pm GMT */
   /* Time for a leap-second!!!                                 */
   nsecs=61 ;
}