/*                                                   Copyright (c) CMI, 2007
 *****************************************************************************
 *  Timer.c
 *
 *  Timer initialization and interrupt service routines
 *  Implicitly in charge of display refresh too
 ****EXTERNAL*****************************************************************
 * ledtimer_init   : initialize the timer 2 interrupts
 * showtime        : command to show the time
 ****INTERNAL*****************************************************************
 * tmr3_isr        : Timer 2 ISR in charge of refreshing the display
 *****************************************************************************
 */
#include <ez80.h>
#include <string.h>

#include "def.h"
#include "globals.h"
#include "timer.h"
#include "time.h"
#include "ledmatrix.h"
#include "transition.h"
#include "syscalls.h"
#include "asmhelp.h"
#include "tables.h"
#include "gpio.h"
#include "wwvb.h"

/* local routines */
void interrupt tmr3_isr( void ); 
 
/* local variables */
UBYTE dmode ;
/* local defines */
#define SYSTEMCLOCK 50000000
#define TMR3_IVECT 0x60

 
/****************************************************************************
*  ledtimer_init(interval)
*
*  initialization routine for timer 3 which will run the LED display refresh.
*  Sets up timer3 for a 1.000ms interval.  The timer is set up such that the
*  OC3 output goes low to disable the drivers and fire off the IRQ.  The 
*  service routine then writes the new data in to the drivers in time for
*  the timer to reach 0x0001 when the new data is driven to the LED matrix.
*  The CPU only gets interrupted on the first compare, and when inside the 
*  interrupt routine, simply polls for the count of 0x0001 before continuing
*  other tasks.
*
****************************************************************************/
void ledtimer_init( void )
{
   UBYTE tmp; /* unused value to clear timer hardware    */
   
   /* start off by disabling timer 3 for reprogramming */
   TMR3_CTL = 0x00;

   /* set Timer 3 interrupt vector */
   set_vector( TMR3_IVECT, tmr3_isr );

   /* set up the timer period */
   /* I want a 1.0000ms period, but a SW-controlled duty cycle */
   /* I'll set the period of the counter to be 1.000ms then the OCs to toggle the LEDs output enables */
   /* The timer counts at 50MHz/16, = 3.125MHz, so divide by 3125 to get 1KHz (0x0c35)*/
   TMR3_RR_H = 0x0c ;
   TMR3_RR_L = 0x35 ;

   /* on to the part-specific stuff to re-enable the timer */
   tmp          = TMR3_IIR ; /* clear out any pending IRQs */
   TMR3_IER     = 0x40     ; /* IRQ on Output Compare 3 (OC3) only */
   TMR3_OC_CTL1 = 0x01     ; /* enable output compares, initialize to 0's, no master mode */
   TMR3_OC_CTL2 = 0x40     ; /* OC3 will go low on compare */
   TMR3_OC3_L   = 0x00     ; /* LSB of compare value = 0 for now (debug) */
   TMR3_OC3_H   = 0x01     ; /* MSB of compare value = 1 for now (debug) */
   
   /* set the display mode to off for now */
   dmode= SHOWTIME_OFF ;

   /* initialize our ticks counter */
   tim50ms=0 ;

   /* make sure the transition engine is dormant */
   transition_busy=FALSE ;

   /* OK things are set up, fire things off! */
   TMR3_CTL = 0x0F; /* Run on breakpoints, system clock/16 source, continuous mode, reload, and enabled */

   /* not sure why the ZDSII 4.7.1 tools didn't need me to enabled interrupts, */
   /* but the 4.11.0 tools certainly do, this took 2+ hours to isolate.        */
   asm (" ei") ;

   }

/****************************************************************************
* Routne   : showtime
* Gazintas : mode (display mode)
* IOs      : None
* Returns  : Nothing
* Globals  : dmode
*
* This routine is called to set the dispay mode.  All I really do here is 
* some simple bounds checking and then save the value.
****************************************************************************/
void showtime ( UBYTE mode )
{
   /* some simple bounds checking */
   if (mode>SHOWTIME_SSSEQ) mode=SHOWTIME_OFF ;

   /* save it! */
   dmode=mode ;  /* for the ISR code      */
   newmin=TRUE ; /* force a time "redraw" */
}

/****************************************************************************
* Routne   : del50ms
* Gazintas : ticks (# of 50us ticks)
*            mode (now or last call)
* IOs      : None
* Returns  : Nothing
* Globals  : dmode
*
* This routine delays a number of ticks of the 50us refesh clock.  Ticks
* is the number of ticks to delay.  Mode is DEL_NOW for starting when this
* routine is called, DEL_LAST for relative to when this routine was called
* last.
* NOTE!!!  This is more useful than it looks as the counter used (tim50ms)
* is synchronized with the display refresh code, so this will give very
* smooth timed effects in sync with the display refresh.
* NOTE!!! Do make sure your first call doesn't use DEL_LAST as you need a
* previous value to start with!
****************************************************************************/
void del50ms ( UBYTE ticks, UBYTE mode )
{
   static UBYTE prev ;

   if (mode==DEL_NOW) prev=tim50ms ;

   while (tim50ms!=(UBYTE)(prev+ticks)) ;

   prev=tim50ms ;

}

/****************************************************************************
*  timer 3 interrupt service routine
*
*  controls the LED Matrix refresh rate, clock updates, and button scanning
*  Round 1 (all C) takes 70.0us from IRQ to poll.  It can be better.
*    After some assembly optimizations, I have it down to 22us!
****************************************************************************/
void interrupt tmr3_isr( void )
{
   UBYTE tmp            ;    /* trash variable             */
   static UBYTE row = 0 ;    /* active row being refreshed */
   static UBYTE seq = 0 ;    /* state machine sequencer    */
   static UBYTE phase  = 0 ; /* Display phase for blinking */
   UBYTE clk_hi,clk_lo ;     /* '595 clock high and low values  */

   PA_DR=wwv_pa|0x82 ; /* flag we are in the routine */

   /* Set up the timer to fire at the end of our updates */
   /* I triger on a count of 0x0001.  A count of 0x0000 doesn't fire??!! */
   tmp          = TMR3_IIR ; /* clear out any pending IRQs */
   TMR3_IER     = 0x00     ; /* No IRQs for now */
   TMR3_OC_CTL2 = 0x80     ; /* OC3 will go high on compare */
   TMR3_OC3_L   = 0x01     ; /* LSB of compare value = 1 */
   TMR3_OC3_H   = 0x00     ; /* MSB of compare value = 0 */

   /* if we are in display hold mode, run silent */
   if (doff)
   {
      PB_DR=0x00 ; /* no rows lit */
   }
   else 
   {
      PA_DR=wwv_pa|0x82; /* make sure the register reset isn't set */

      /* the clock high and low values need to account for the WWVB enable */
      /* bit, so compute those ahead of time before we use them.           */
      clk_hi=wwv_pa|0x86 ;
      clk_lo=wwv_pa|0x82 ;

      if (phase==0)
      {
         /* send out the clock data (asm way)        */
         /* Could not get C way to be very efficient */
         send_clk(clkmtxa[row  ]) ;
         send_clk(clkmtxa[row+8]) ;

         /* now send the matrix data (C way)*/
         /* This is actually quite efficient, may leave alone! */
         PC_DR=dspmtxa[row*4];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxa[row*4+1];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxa[row*4+2];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxa[row*4+3];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxa[row*4+32];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxa[row*4+33];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxa[row*4+34];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxa[row*4+35];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
      }
      else /* phase B */
      {
         /* send out the clock data (asm way)        */
         /* Could not get C way to be very efficient */
         send_clk(clkmtxb[row  ]) ;
         send_clk(clkmtxb[row+8]) ;

         /* now send the matrix data (C way)*/
         /* This is actually quite efficient, may leave alone! */
         PC_DR=dspmtxb[row*4];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxb[row*4+1];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxb[row*4+2];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxb[row*4+3];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxb[row*4+32];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxb[row*4+33];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxb[row*4+34];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
         PC_DR=dspmtxb[row*4+35];
         PA_DR=clk_hi;
         PA_DR=clk_lo;
      }

      /* data sent, clock it from shift registers to output latches */
      PA_DR=wwv_pa|0x83;
      PA_DR=wwv_pa|0x82;

      /* assert the correct row driver */
      PB_DR=oneofeight[row] ;
   } ;

   PA_DR=wwv_pa|0x02 ; /* flag we waiting for the timer */
   /* Inside the IRQ, wait for OC3 to fire using polling */
   while (!(TMR3_IIR&&0x40)) ;   
   PA_DR=wwv_pa|0x82 ; /* flag the timer fired */

   /* Now set up timer and IRQ to fire after this row refresh cycle */
   tmp          = TMR3_IIR ; /* clear out any pending IRQs */
   TMR3_IER     = 0x40     ; /* IRQ on Output Compare 3 (OC3) only */
   TMR3_OC_CTL2 = 0x40     ; /* OC3 will go low on compare */
   TMR3_OC3_L   = 0x00     ; /* LSB of compare value = 0 for now (debug) */
   TMR3_OC3_H   = 0x01     ; /* MSB of compare value = 1 for now (debug) */

   /* increment and bound our row counter */
   row=(row+1)&0x07 ;

   /**********************************************************************/
   /* done with the display refreshing part, so now on to the sequencer  */
   /* This sequencer counts from 0-49 on every IRQ, so loops every 50ms. */
   /* 1) In that time we need to increment the 60ths of a second three   */
   /*    times and update the display to reflect that.                   */
   /* 2) Need to scan the keys once.                                     */
   /* 3) Call the transition engine (wipes, etc.) twice                  */
   /**********************************************************************/
   /* Sequencer state assignments are as follows:                        */
   /* #1)  Go read the buttons                                           */
   /* #2)  Transition engine update #1                                   */
   /* #4)  Go run the WWVB state machine                                 */
   /* #6)  Check for leap seconds                                        */
   /* #14) Update time (60th of a second)                                */
   /* #16) Update display (for above)                                    */
   /* #27) Transition engine update #2                                   */
   /* #31) Update time (60th of a second)                                */
   /* #33) Update display (for above)                                    */
   /* #47) Update time (60th of a second)                                */
   /* #49) Update display (for above)                                    */
   /* If none of above *and* matrix update, go render it.                */
   /**********************************************************************/

   switch (seq)
   {
      case  1 : gpio_rdkeys()      ; break ;
      case  2 : transition_engine(); break ;
	  case  4 : wwvbsm()           ; break ;
	  case  6 : wwvlsc()           ; break ;
      case 14 : time_inc()         ; break ; 
      case 16 : time_dsp(dmode)    ; break ; 
      case 27 : transition_engine(); break ;
	  case 31 : time_inc()         ; break ;
	  case 33 : time_dsp(dmode)    ; break ;
	  case 47 : time_inc()         ; break ;
	  case 49 : time_dsp(dmode)    ; break ;
	  default : if (newmin)
	            {
				   newmin=FALSE ;
				   ledmatrix_wrtime(dmode,TRT_FANCY) ;
				}
                break ;
	} ;

	/* increment the sequencer and change phases every 50ms */
	seq++ ;
	if (seq>=50)
    {
       seq=0 ;
	   tim50ms++ ;
	   if (phase==0)
       {
          phase=1 ;
	   }
	   else
       {
	      phase=0 ;
	   }
	}

    /* flag we are done with the IRQ */
    PA_DR=wwv_pa|0x02 ;
}

