/*                                         Copyright (c) CMI Productions, 2007
 *****************************************************************************
 *  menus.c (1.0)
 *
 *  Menu functions 
 ****EXTERNAL*****************************************************************
 * menus   : User Setup Menus
 * get_tzo : Get timezone offset
 * get_dso : Get Daylight Savings Observance
 *
 * This is the main menu routine that gets called when the user presses the
 * menu button when the time is pressed.  It goes through the menu sequences:
 *  SET WWVB : to set WWVB parameters
 *  GET WWVB : to get the time from WWVB
 *  SET TIME : to set the time manually
 ****INTERNAL*****************************************************************
 * None
 *****************************************************************************
*/

#include <ez80.h>
#include <string.h>

#include "def.h"
#include "gpio.h"
#include "time.h"
#include "globals.h"
#include "LedMatrix.h"
#include "transition.h"
#include "timer.h"
#include "tables.h"
#include "flash.h"
#include "wwvb.h"
#include "mystring.h"

/* Local Routines */
void set_wwvb  (void) ;
void time_last (void) ;

/****************************************************************************
* Routne   : menus
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : None (directly)
*
* This is the routine we go to when the menu button is pressed while
* displaying the time.  The menu button cycles through the various menus
* until the user either presses enter (to to execute that routine) or they
* sequence through all the menus and end up back at showing the time.
****************************************************************************/
void menus ( void )
{
   UBYTE key=KEY_NONE ; /* key being pressed */
   UBYTE menu=0       ; /* Menu number       */

   /* turn off the time display for a bit */
   showtime(SHOWTIME_OFF) ;

   /* four menus, so cycle until done */
   while ((menu<4)&&(key!=KEY_ENTER))
   {
      ledmatrix_clear(ACTIVE_PLANE) ;
      switch(menu)
	  {
	     case 0 : /* write out SET WWVB */
                  ledmatrix_puts(6,0,(UBYTE*)"SET" ,PHASEAB,ACTIVE_PLANE) ;
                  ledmatrix_puts(3,8,(UBYTE*)"WWVB",PHASEAB,ACTIVE_PLANE) ;
				  break ;
	     case 1 : /* write out SET WWVB */
                  ledmatrix_puts(6,0,(UBYTE*)"GET" ,PHASEAB,ACTIVE_PLANE) ;
                  ledmatrix_puts(3,8,(UBYTE*)"WWVB",PHASEAB,ACTIVE_PLANE) ;
				  break ;
	     case 2 : /* write out SET WWVB */
                  ledmatrix_puts(3,0,(UBYTE*)"LAST",PHASEAB,ACTIVE_PLANE) ;
                  ledmatrix_puts(3,8,(UBYTE*)"WWVB",PHASEAB,ACTIVE_PLANE) ;
				  break ;
	     case 3 : /* write out SET WWVB */
                  ledmatrix_puts(6,0,(UBYTE*)"SET" ,PHASEAB,ACTIVE_PLANE) ;
                  ledmatrix_puts(3,8,(UBYTE*)"TIME",PHASEAB,ACTIVE_PLANE) ;
				  break ;
         default: break ; /* should never happen */
      }

      /* got here by user pressing the menu key */
      /* wait for the next key to be pressed    */
      key=gpio_nextkey() ;

      /* if key pressed is Menu, on to next menu */
      if (key==KEY_MENU)
      {
         menu++ ;

      }
	  else if (key==KEY_ENTER)
	  {
         switch(menu)
         {
	        case 0 : set_wwvb()  ; break ;
	        case 1 : time_get()  ; break ;
	        case 2 : time_last() ; break ;
	        case 3 : time_set()  ; break ;
            default: break ; /* should never happen */
         }
	  }
   } /* while */

   /* all done, wait for no keypress then return */
   gpio_nokey() ;
}

/****************************************************************************
* Routne   : set_wwvb
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : timezone, DST
*
* This routine sets up the various NVM variables needed for correctly 
* converting WWVB time to local time. These include:
*  + timezone - offset from UTC to local
*  + observe_DST - Flag to follow DST or not
*
****************************************************************************/
void set_wwvb (void) 
{
   UBYTE tzp ; /* Timezone index */
   UBYTE dso ; /* DST flag       */
   UBYTE key ; /* key value      */

   /* read the time zone and DSO from flash     */
   /* if tzp is 0xff (erased), leave default    */
   /* if dso is 0xff assume that is TRUE        */
   tzp=read_data_from_infopage(0,0) ;
   dso=read_data_from_infopage(0,1) ;
   if (tzp==0xff) tzp=3 ;

   /* first take care of the time zone */
   do
   {
      /* write out the selected time zone */
      ledmatrix_clear(ACTIVE_PLANE) ;
      ledmatrix_puts(0,0,(UBYTE*)"TZONE",PHASEAB,ACTIVE_PLANE) ;
      ledmatrix_puts(tz_table[tzp].x,tz_table[tzp].y,tz_table[tzp].text,PHASEAB,ACTIVE_PLANE) ;

      /* wait for the next key to be pressed    */
      key=gpio_nextkey() ;

      /* process up and down keys (menu is ignored) */
      if (key==KEY_UP)   tzp++ ;
	  if (key==KEY_DOWN) tzp-- ;

	  /* bounds checking */
	  if (tzp==7) tzp=0 ;
	  if (tzp>7)  tzp=6 ; /* funky bounds check for -1 with unsigned char */

   } while (key!=KEY_ENTER) ;

   /* now do DST observation */
   do
   {
      /* write out the selected DST observation */
      ledmatrix_clear(ACTIVE_PLANE) ;
      ledmatrix_puts(3,0,(UBYTE*)"DST?" ,PHASEAB,ACTIVE_PLANE) ;
      if (dso)
	  {
         ledmatrix_puts(6,8,(UBYTE*)"YES"  ,PHASEAB,ACTIVE_PLANE) ;
	  }
	  else
	  {
         ledmatrix_puts(9,8,(UBYTE*)"NO"   ,PHASEAB,ACTIVE_PLANE) ;
	  }

      /* wait for the next key to be pressed    */
      key=gpio_nextkey() ;

      /* Up or Down toggles DSO (menu is ignored) */
      if ((key==KEY_UP)||(key==KEY_DOWN)) dso=!dso ;

   } while (key!=KEY_ENTER) ;


   /* With changes made, see if anything needs rewriting to the information page */
   /* Compare new values to those in info page to see if any differences.        */
   if ((tzp!=read_data_from_infopage(0,0))||
       (dso!=read_data_from_infopage(0,1)))
   {
      unlock_flash()                  ; /* unlock the flash    */
      erase_info_page ()              ; /* erase info page     */
      write_byte_to_infopage(tzp,0,0) ; /* write the time zone */
      write_byte_to_infopage(dso,0,1) ; /* write DS Observance */
      lock_flash()                    ; /* relock the flash    */
   }

   /* got here by user pressing the menu key */
   /* wait for the next key to be pressed    */
   gpio_nokey() ;

}

/****************************************************************************
* Routne   : get_tzo
* Gazintas : None
* IOs      : None
* Returns  : Time Zone offset
* Globals  : info page in flash
*
* This routine returns the timezone offset as set by the user.
*
****************************************************************************/
BYTE get_tzo (void)
{
   UBYTE tzp ;

   tzp=read_data_from_infopage(0,0) ; /* get the index from info page */
   if (tzp==0xff) tzp=3             ; /* if unprogrammed, assume MST  */
   return (tz_table[tzp].offset)    ; /* return offset from table     */
}

/****************************************************************************
* Routne   : get_dso
* Gazintas : None
* IOs      : None
* Returns  : Daylight Savings Observance Flag
* Globals  : info page in flash
*
* This routine returns the Daylight Savings Observance flag as set by the
* user.  Weirdly, since it is a boolean, no bounds checking needed!
*
****************************************************************************/
BYTE get_dso (void)
{
   UBYTE tzp ;

   return(read_data_from_infopage(0,1)) ; /* just return byte from flash! */
}


/****************************************************************************
* Routne   : time_last
* Gazintas : None
* IOs      : None
* Returns  : Daylight Savings Observance Flag
* Globals  : ww_hrs, ww_mins, ww_month, ww_day (through a procedure call)
*
* This routine gets the last time received by the WWVB routines and displays
* it for a few seconds.  Idea is to show the last valid time received.
* The display requirements are different enough (unfortunately) to not share
* this code with the regular time-showing code.
*
****************************************************************************/
void time_last (void)
{
   UBYTE last_mins  ; /* last ACQ Minutes */
   UBYTE last_hours ; /* last ACQ Hours   */
   UBYTE last_day   ; /* last ACQ Day     */
   UBYTE last_month ; /* last ACQ Month   */
   UBYTE str[6]     ; /* the place to put the date/time */

   ledmatrix_clear(ACTIVE_PLANE) ;

   /* ask the WWVB code when the last acquisition was */
   wwvb_last(&last_month,&last_day,&last_hours,&last_mins) ;

   itoa2(last_hours,str)  ;  /* hours in 1st two bytes  */
   str[2]=':'             ;  /* the semicolon           */
   itoa2(last_mins,str+3) ;  /* write the minites       */
   str[5]=0               ;  /* terminate the string    */

   /* show the time in 24 hour format */
   ledmatrix_puts(0,0,str,PHASEAB,ACTIVE_PLANE) ;

   /* now on to the date */
   itoa2(last_month,str) ;  /* wrute the month         */
   str[2]='/'            ;  /* the semicolon           */
   itoa2(last_day,str+3) ;  /* write the day           */
   str[5]=0              ;  /* terminate the string    */
   ledmatrix_puts(0,8,str,PHASEAB,ACTIVE_PLANE) ;

   /* wait three seconds... */
   del50ms(60,DEL_NOW) ;

}
