        page
******************************************************
** Copyright 2007 CMI Productions.                  **
** Lots of rights reserved.                         **
** Free for personal use as long as this copyright  **
** message is maintained in the source and the LCD  **
** copyright message is also preserved.  Commercial **
** use prohibited without CMIs written concent.     **
******************************************************
; -----------------------------------------------------------------------------
; File:         timirq.asm
; Function:     Timer setup and interrupt service handler
; Routines:     initmr: Initialize the timer hardware
;               timirq: general timer interrupt handler
; -----------------------------------------------------------------------------

;       The timer setup and interrupt routines.
;       These routines make sure that the timer interrupt gets called every
;       1.000ms.  The timer routine is in charge of refreshing the LED matrix,
;       so any timing inaccuracies will show up as display flickering.  SW
;       inaccuracies are minimized by using a single "blanking" line to the
;       column drivers and having that blanking line controlled by the 6808
;       timers.  SW routines to set up the LED states are updated during the
;       blanking time, so even though that timing may be variable, the actual
;       on-time of the LEDs should be crystal-accurate.
;
;       The bus clock is either 2.4576MHz (debug mode) or 8.000MHz (normal
;       mode).  To get to our 1.000KHz refresh rate we don't need any
;       prescaling, so divisors can be 2458($099a) or 8000($1f40) respectively.
;
;       Here is where we scan the keypad too.  We read the buttons
;       every 100ms.  Since we sample data instead of looking for
;       edges, we don't need to do debouncing.
;
;       Here we also increment our real-time clock.  We have nine
;       counters to keep track of: year, month, day,
;       local bours, hours, minutes, seconds, and
;       sub-seconds (60ths).  We also do not increment the time
;       when the rtchld semaphore is set.  It is set when we are
;       messing with setting the time.
;
;       As the base routine needs to be executed every 1ms for the display
;       refresh, that code must be very compact and tight.  It will fire off
;       other real-time, but lesser critical tasks.  These tasks include:
;       1) Keyboard scanning
;       2) Real-time clock keeping

;
        page
; -----------------------------------------------------------------------------
; Routine:      initmr
; Called By:    main
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Function:     This initializes the timer hardware.  While it sounds weird,
;               I do not use the PWM mode of the timer for running the display.
;               instead I implement the PWM in softare.  Reason is that I
;               *WANT* the timer functions to stop if the CPU stops.  If I
;               don't, then I could blow up LEDs by driving them more than 1/8th
;               duty cycle (the CPU is in charge of sequencing row drivers)
;               Hence, for the timer setup I just tell it to go count bus
;               cycles at the bus frequency without any prescaling or overflow
;               interrupts.  All of the wiggling of IO lines is done by the
;               timer compare registers under software control
;               In Debug mode, bus frequency is 2.4576MHz.
;               In Normal mode, bus frequency is 8.0000MHz.
; -----------------------------------------------------------------------------
initmr: clr     t1ch0h          ; Clear out timer 1 chan 0 to 0000
        clr     t1ch0l          ; done

        mov     #$30,t1sc       ; Stop the timer and assert the reset bit
        mov     #$54,t1sc0      ; Output compare mode, generate IRQ, toggle output
        mov     #$00,t1sc       ; No Overflow irq, start, no prescaling
        cli                     ; enable interrupts!!!
        rts                     ; done

        page
; -----------------------------------------------------------------------------
; Routine:      timrtn (basics)
; Called By:    Timer Interrupt
; Calls:        timrtc - Update real time clock
;               timkey - scan keyboard
; Gazintas:     clkmtx - Clock led matrix
;               dspmtx - Display LED matrix
; Gazoutas:     The display is refreshed!
;               ruftim - 1ms counter, counts from 0-99 then resets)
; Function:     This is the timer interrupt service routine.  As it is called
;               every 1ms, I need to keep it as short and predictable possible.
;               Main purpose is to refresh the LED matrix display.  I also
;               keep a counter (ruftim) that counts to 50 and cycles.  I then
;               fire of some routines based on that count.  Of particular
;               importance is a routine fired off every 17,17,16 cycles
;               (average 16 and 2/3rds) which gets us our 1/60th of a second
;               counter for the RTC.  Other routines are the keyboard scanning
;               and time drawing rouintes (more to come).
;               !!!WARNING!!!  At the end of this routine, if we do end up
;               calling the matrix update routine, I do re-enable interrupts.
;               before calling that routine.  That code can take longer than
;               1ms to execute, so I need to enable interrupts to keep the
;               display refresh solid.  I am essentially allowing for nested
;               interrupts, not something the 6808 is intended to do!
; -----------------------------------------------------------------------------
;
; First figure out the timer constants depending on debug mode or not
;
$IF DEBUG
timper: equ     2458            ; Bus cycles in 1ms
timpon: equ     135+135+67      ; IRQ service time (135us) times 2.5
timpof: equ     2458-timpon     ; Rest of the time
#ELSEIF
timper: equ     8000            ; Bus cycles in 1ms
timpon: equ     135*8           ; IRQ service time (135us) times 8
timpof: equ     8000-timpon     ; Rest of the time
$ENDIF


timrtn: pshh                    ; save H, this is an IRQ routine
;       bset    1,portc         ; flag inside irq (for debug/fine tuning only)
        lda     t1sc0           ; read status register with IRQ bit set so clr will work

;       First thing to do is tell the timer when to turn the LED matrix back
;       on with the new data.  The delay is short and intended to be slightly
;       longer than this IRQ routine takes.  I wait in here for this next
;       compare before exiting the IRQ routine

        lda     t1ch0l          ; get current ocr LSB
        add     #timpon%256     ; add LSB of on-time
        tax                     ; hold on to that answer for a bit
        lda     t1ch0h          ; get current MSB
        adc     #timpon\256     ; add MSB of on-time (with carry)
        sta     t1ch0h          ; store MSB
        stx     t1ch0l          ; store LSB
        mov     #$1c,t1sc0      ; clear IRQ, no IRQ next, toggle low

;       Next is drive the correct row driver.  For rows 0-7 we sequence through
;       four bits each on port a and port d.  This is done with a lookup table.

        ldx     drow            ; get display row number
        clrh                    ; HX is index
        lda     rowtb1,x        ; get port A row driver pattern
        sta     porta           ; assert it
        lda     rowtb2,x        ; get port D row driver pattern
        sta     portd           ; assert it

;       Now, depending on the phase, I select either Phase A (clkmtx) or Phase B
;       (clkmty).  Rather than do math for each fetch, I just increment x (the
;       row offst) by 16 if we are in Phase B and not if we are in Phase A.

        tst     phase           ; check the phase
        beq     timshf          ; if Phase A, nothing to do
        txa                     ; Get X to A for addition
        add     #16             ; offset to Phase B
        tax                     ; and put back in X

;       Now send out the radial clock LED matrix bits.  I only send the two
;       LSBs of the data to the shift registers so need to read a byte then
;       do write/shift operations.  In the interests of speed, I forgo a loop.
;       As the first write is the minutes table and the second write is for the
;       hours table, I parse the clkmtx table in two halves

                                ; X is still row. Still x1 as I parse two tables
timshf: clrh                    ; HX is now offset into clock LED array
        lda     clkmtx,x        ; get the second table byte
        sta     portb           ; drive two LSBs
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        lsr     portb           ; shift down two bits
        lsr     portb           ; shift down two bits
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        lsr     portb           ; shift down two bits
        lsr     portb           ; shift down two bits
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        lsr     portb           ; shift down two bits
        lsr     portb           ; shift down two bits
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        lda     clkmtx+8,x      ; get the hours table byte

        sta     portb           ; drive two LSBs
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        lsr     portb           ; shift down two bits
        lsr     portb           ; shift down two bits
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        lsr     portb           ; shift down two bits
        lsr     portb           ; shift down two bits
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        lsr     portb           ; shift down two bits
        lsr     portb           ; shift down two bits
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

;       Now on to the big LED matrix in the center of the clock.  Each row
;       is an incremental 4 bytes from the display matrix array.  The way
;       the shift registers are set up, the rows are split in two, so the
;       first "row" is actually rows 1 and 5, second is 2 and 6...
;       HX is still the row number, so I just need to multiply it by four
;       to get the row offset then add 32 to get the second row.

        ldx     drow            ; get display row number back (without offset)
        lslx                    ; x is now row x2
        lslx                    ; x is now row x4 (H still zero)
        txa                     ; put in A
        add     #dspmtx         ; add display matrix base (still less than 0xff)
        tax                     ; X is now pointer to bytes to send

        mov     x+,portb        ; a display byte
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        mov     x+,portb        ; a display byte
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        mov     x+,portb        ; a display byte
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        mov     x+,portb        ; a display byte
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        aix     #28             ; on to the bottom half

        mov     x+,portb        ; a display byte
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        mov     x+,portb        ; a display byte
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        mov     x+,portb        ; a display byte
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

        mov     x+,portb        ; a display byte
        bset    1,porta         ; wiggle shift register clock high
        bclr    1,porta         ; wiggle shift register clock low

;       All the data has been shifted into the shift registers, so all that
;       is left to do is clock it into the output registers

        bset    3,porta         ; wiggle output register clock high
        bclr    3,porta         ; wiggle output register clock low
;       bclr    1,portc         ; say we're done
;       That's it, now wait for the timer that enables the LED drivers to
;       fire.

timwt:  brclr   7,t1sc0,timwt   ; wait for the compare flag
;       bset    1,portc         ; say we're back to computing

        lda     t1ch0l          ; get current ocr LSB
        add     #timpof%256     ; add LSB of on-time
        tax                     ; hold on to that answer for a bit
        lda     t1ch0h          ; get current MSB
        adc     #timpof\256     ; add MSB of on-time (with carry)
        sta     t1ch0h          ; store MSB
        stx     t1ch0l          ; store LSB
        mov     #$58,t1sc0      ; clear IRQ, IRQ next, toggle high

        lda     drow            ; get the curent row
        inca                    ; increment
        and     #$07            ; constrain to 0-7
        sta     drow            ; save it

;       ruftim is a timer that counts every 1ms to 50 then starts over again.
;       This is used as my sequencer to perform the tasks that need performing
;       on a less frequent basis.  Also, this is where I toggle the phase for
;       the display.  I do it every 50ms for a 100ms period.

        inc     ruftim          ; increment the rough time
        lda     ruftim          ; get rough time counter back
        cmp     #50             ; see if we got to 50
        bne     timseq          ; if not, go do sequencing
        clr     ruftim          ; otherwise start over at 0
        lda     phase           ; get the current phase
        eor     #1              ; toggle LSBit
        and     #1              ; keep it 0 or 1
        sta     phase           ; and save it

;       This is sequencer that lets me perform tasks partitioned out to steps
;       every ms that repeat every 50ms.  Note that the routines called by the
;       cbeqa branches below return to the timdun label, not to the caller.
;       General idea is to break big tasks into littler tasks that perform in
;       the background.  No task should exceed 1ms or bad things happen.
;       Exception dttim below.  That does take more than 1ms so I make sure the
;       basic timer stuff is done, then re-enable interrupts so that a timer
;       interrupt can interrupt the dttim routine.  Dangerous stuff, but works
;       OK if handled with care.

timseq: lda     ruftim          ; increment rough time counter
        beq     timkey          ; if 0, go do keyboard code and blink code

        cbeqa   #14,timrtc      ; 1st of 3 RTC ticks per 50-count
        cbeqa   #31,timrtc      ; 2nd of 3 RTC ticks per 50-count
        cbeqa   #47,timrtc      ; 3rd of 3 RTC ticks per 50-count

;       Leave an unused gap between these three pairs of calls (#15, 32, 48)

        cbeqa   #16,timdsp      ; 1st of 3 calls to update display
        cbeqa   #33,timdsp      ; 2nd of 3 calls to update display
        cbeqa   #49,timdsp      ; 3rd of 3 calls to update display

;       Once per 50ms loop, call the WWVB parsing routine
        cbeqa   #2,timwwv       ; go check the WWVB state machine

;       This isn't terribly efficient yet, but once per 50ms, check for leap seconds
        cbeqa   #4,timlsj       ; go check for leap seconds

;       This snippet only gets executed if none of the above regularly scheduled
;       tasks are scheduled.  This is the code to update the text portion of the
;       display.  The newtim flag gets set if there is new data to show.  That
;       happens if either we get a new minute or anything changes during the
;       time set routine.  Note that due to the length of time this routine takes
;       to execute (>1ms), I need to re-enable interrupts before calling the
;       routine.  This is OK as we've done all the needed display stuff first, so
;       we are effectively re-entrant

        tst     dsphld          ; first see if display updates are on hold
        bne     timdun          ; if so, don't mess with displaying the time
        tst     newtim          ; if none of the above, check to see if new time
        beq     timdun          ; if not, all done
        cli                     ; Enable interrupts! (see two text boxes above)
        jsr     dttim           ; go update the time

timdun: ;bclr     1,portc        ; flag done with irq
        pulh                    ; restore H
        rti                     ; all done with interrupt

; -----------------------------------------------------------------------------
;       Need a jsr instead of a branch to dtime
; -----------------------------------------------------------------------------
timdsp: tst     dsphld          ; see if we are supposed to show time
        bne     timdun          ; if non-0, don't write the time
        jsr     dtime           ; go render the time
        bra     timdun          ; and go back to the display loop

; -----------------------------------------------------------------------------
;       Need a jsr instead of a branch to wwvbsm
; -----------------------------------------------------------------------------
timwwv: jsr     wwvbsm          ; if so, go execute the WWVB state machine
        bra     timdun          ; and go back to the display loop

; -----------------------------------------------------------------------------
;       Need a jsr instead of a branch to wwvlsc
; -----------------------------------------------------------------------------
timlsj: jsr     timlsc          ; if so, go execute the WWVB state machine
        bra     timdun          ; and go back to the display loop

        page
; -----------------------------------------------------------------------------
; Routine:      timkey (Timer keyscan section)
; Called By:    Timer Interrupt
; Calls:        None
; Gazintas:     Nothing
; Gazoutas:     curkey
; Function:     This is the keyboard scanning code.  It gets called once every
;               50ms by the main timer routine.  By scanning only once every
;               50ms, there isn't much need for debouncing.
;               NOTE, this is not a subroutine, it jumps back to the blink
;               code which finally jumps back to the main routine.
; -----------------------------------------------------------------------------
timkey: lda     portc           ; get the key bits (0,2,3,4)
        tax                     ; save it for a bit
        lsra                    ; Bits 2-4 in 1-3
        lsra                    ; Bits 2-4 in 0-2
        lsrx                    ; Put bit 0 in carry
        rola                    ; Bits now 0,2,3,4
        and     #$0f            ; only 4 LSBs for table
        tax                     ; make an index
        clrh                    ; make sure H is 0
        lda     keytbl,x        ; get keycode
        sta     curkey          ; store it
        bra     timblk          ; all done, go to blink routine

        page
; -----------------------------------------------------------------------------
; Routine:      timblk (Blinker control section)
; Called By:    Timer Interrupt
; Calls:        None
; Gazintas:     Nothing
; Gazoutas:     blink: blink control byte
;               newtim: flag to tell dttim to redraw the time
; Function:     This routine wiggles the MSB of the blink variable every 50
;               ms (every time it is called).  If any of the control fields
;               (bits 0-6) are set, the newtim flag is also set to tell
;               the dttim routine to re-render the time.
;               NOTE, this is not a subroutine, it jumps back to the main
;               routine.
; -----------------------------------------------------------------------------
timblk: lda     blink           ; get the blink state
        eor     #$80            ; wiggle the MSB
        sta     blink           ; save it
        and     #$7f            ; look only at the control bits now
        beq     timbld          ; if none set, don't need to re-render
        mov     #1,newtim       ; otherwise tell renderer to re-draw the time
timbld: bra     timdun          ; done, go back to timer code

        page
; -----------------------------------------------------------------------------
; Routine:      timrtc (Timer real-time-clock section)
; Called By:    Timer Interrupt
; Calls:        mthmax - figgers last day of specified month)
; Gazintas:     rtchld (0=running, 1=hold)
;               wwvdsf (DST change flag)
;               nsecs # of seconds in current minute (for leap seconds)
; Gazoutas:     ssecs,secs,mins,hrs,day,month,year (The time)
; Function:     This is the real-time clock support code.  Basically everything
;               gets incremente by a 60th of a second for each call.
;               If rtchld is asserted, then this code gets skipped.  It is
;               asserted whenever the time setting routine is running.
;               Theoretically this bugger is Y2K-ready.
;               NOTE, this is not a subroutine, it jumps back to the timer code!
;               Besdes normal timekeeping, I also have to handle DST transitions
;               as well as leap seconds.  Detailed support is in the comments
;               with the code.
; -----------------------------------------------------------------------------
timrtc: tst     rtchld          ; see if RTC on hold
        beq     timrun          ; if 0, do the time thing
        jmp     clkdun          ; if non-0, skip the time thing

timrun: inc     ssecs           ; increment sub-seconds
        lda     ssecs           ; get incremented value
        cmp     #60             ; see if 60 60ths
        blo     clkdun          ; if not, done
        clr     ssecs           ; otherwise clear sub-seconds

        inc     secs            ; increment seconds
        lda     secs            ; get incremented value
        cmp     nsecs           ; see if 60 seconds (or 61 if leap-second)
        blo     clkdun          ; if not, done
        clr     secs            ; otherwise clear seconds

        mov     #1,newtim       ; flag a new minute (for LED matrix update)
        inc     mins            ; increment minutes
        lda     mins            ; get incremented value
        cmp     #60             ; see if 60 minutes
        blo     clkdun          ; if not, done
        clr     mins            ; otherwise clear minutes

;       Here is where I take care of DST transitions, when the hours increment.
;       The wwvdsf flag tells me what to do.
;       0: do nuthin
;       1: change to Standard time (fall back)
;       2: change to DST (spring forward)
;       To prevent repeats, the flag is cleared once "used"

        inc     hrs             ; increment hours
        lda     hrs             ; get incremented value
        cmp     #2              ; see if we just hit 2am
        bne     timn2a          ; if not process normally
        lda     wwvdsf          ; get the DST transition flag
        beq     timn2a          ; if zero, nuthin to do
        cmp     #1              ; see if transition back to standard time
        beq     tim2st          ; if so, go decrement hour back
        inc     hrs             ; if anything else, DST transition, so increment again
        bra     timccf          ; and go clear the flag
tim2st: dec     hrs             ; if to ST, decrement hours back again
timccf: clr     wwvdsf          ; and clear the flag

timn2a: lda     hrs             ; get incremented value again
        cmp     #24             ; see if 24 hours
        blo     clkdun          ; if not, done with UTC, on to local
        clr     hrs             ; otherwise clear hours

        mov     #1,wwvbst       ; at local midnight, fire off the WWVB state machine!

        inc     day             ; increment day-of-month
        lda     month           ; get month
        ldx     year            ; get year
        jsr     mthmax          ; get month max value in A

        cmp     day             ; compare to current day count
        bhs     clkdun          ; if days-in-month higher, then no action
        lda     #1              ; new day is 1st of month
        sta     day             ; remember that

        inc     month           ; increment month
        lda     month           ; get incremented value
        cmp     #13             ; see if December
        blo     clkdun          ; if not, done
        lda     #1              ; new month is January
        sta     month           ; remember that

        inc     year            ; increment year
        lda     year            ; get incremented value
        cmp     #100            ; see if 100 year (Y2K!)
        blo     clkdun          ; if not, done
        clr     year            ; otherwise clear year to 00
clkdun: jmp     timdun          ; done

        page
; -----------------------------------------------------------------------------
; Routine:      mthmax
; Called By:    timirq, stime
; Calls:        None.
; Gazintas:     a=month (the current month)
;               x=year (the current year)
;               mthtbl (table of days in each month)
; Gazoutas:     A: the number of days in the month
; Function:     This routine figgers out the number of days in the current
;               month. For most months, this is a simple lookup of the
;               current month.  For February I need to keep track of leap
;               years.  Fortunately, Feb 2000 is a normal leapyear, so I don't
;               have to make special provisions. Good 'til 2199 I think.
; -----------------------------------------------------------------------------
mthmax: cmp     #2              ; see if February
        bne     mnmaxm          ; if not Feb, normal processing
        psha                    ; save month
        txa                     ; if feb, get year (in X) in A
        and     #$03            ; look at two LSBs of year only
        pula                    ; get month back in A
        bne     mnmaxm          ; if not 00 (4,8,12...) regular year
        lda     #29             ; Only thing left are leapyears
        bra     mnmaxx          ; go figger it

mnmaxm: tax                     ; put month in X
        clrh                    ; make sure H is 0
        decx                    ; convert 1-12 to 0-11
        lda     mthtbl,x        ; get days in the month

mnmaxx: rts                     ; A has month max value

        page
; -----------------------------------------------------------------------------
; Routine:      timlsc
; Called By:    timirq
; Calls:        mthmax
; Gazintas:     secs,mins,hrs,day,month (The time)
;               wwvlsf : Leap second flag
; Gazoutas:
; Function:     This routine figgers out if we need to implement a leap second
;               or not.  Leap seconds extend the last second of the day in a
;               month which turns out hard to do efficiently inside the time
;               incrementing rouinte, so instead I do it as a seperate routine
;               that runs less often.  I check in order of things likely to
;               disqualify the event as a leap second to spend the least time
;               in this routine as possible.  The output of this routine is
;               the number to compare seconds to in order to determine if we
;               should increment the minutes.  This is normally 60 except for
;               when there is a leap second, in which case it is 61.
;               A leap second happens when:
;                - The wwvlsf flag is set (not 0)
;                - It is the last day of the month
;                - It is 11:59pm UTC
; -----------------------------------------------------------------------------
timlsc: mov     #60,nsecs       ; default number of seconds in this minute

        tst     wwvlsf          ; check leap second flag
        beq     timlsx          ; if clear, no leap seconds!

        lda     month           ; get month
        ldx     year            ; get year
        jsr     mthmax          ; get days in current month
        cmp     day             ; see if that is today's day
        bne     timlsx          ; if not, no leap second yet

        lda     hrs             ; get local hours
        ldx     fitz            ; get timezone index
        clrh                    ; h=0...
        sub     tztbl,x         ; subtract local offset to get back to UTC
        cmp     #23             ; see if 11pm
        bne     timlsx          ; if not, no leap second yet

        lda     mins            ; get minutes
        cmp     #59             ; see if 11:59pm UTC
        bne     timlsx          ; if not, no leap second yet

        mov     #61,nsecs       ; if all that passes, leap-second time!

timlsx: rts                     ; all done, exit

