        page
******************************************************
** Copyright 2007 CMI Productions.                  **
** Lots of rights reserved.                         **
** Free for personal use as long as this copyright  **
** message is maintained in the source and the LCD  **
** copyright message is also preserved.  Commercial **
** use prohibited without CMIs written concent.     **
******************************************************
; -----------------------------------------------------------------------------
; File:         dtext.asm
; Function:     Routines for displaying the text in the inner LED matrix
; Routines:     dclr: Clear the display
;               dton: All LEDs on
;               dtest: Simple test pattern (for debug)
;               dputc: Display a single character
;               dputs: Display a string (5 chars max)
;               dpmsg: Displays a string from Flash
;               dttim: Display the time and date (and year if setting it)
; Globals:      dspmtx: The LED display matrix for the three rings of LEDs
; -----------------------------------------------------------------------------

;       The rendering routines for the inner LED matrix.
;       This file is responsible for the routines that fill out the dspmtx array
;       with the right data to display text in the center of the display.
;       The details of that array structure are in the file "LED Matrix.txt".
;       These routines do not do the actual display refresh, that is done in
;       timirq.asm.
;
        page
; -----------------------------------------------------------------------------
; Routine:      dclr
; Called By:    main
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      dspmtx: The LED display matrix for the text display area
; Function:     This just clears the display
; -----------------------------------------------------------------------------
dclr:   ldx     #63             ; wanna clear 64 bytes
        clrh                    ; make sure H is zero
dclrlp: clr     dspmtx,x        ; clear first byte
        dbnzx   dclrlp          ; clear 'em all
        clr     dspmtx          ; clear last one!
        rts                     ; all done

; -----------------------------------------------------------------------------
; Routine:      dclr1
; Called By:    main
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      dspmtx: The LED display matrix for the text display area
; Function:     This just clears line 2 of the display
; -----------------------------------------------------------------------------
dclr1:  ldx     #31             ; wanna clear 32 bytes
        clrh                    ; make sure H is zero
dcl1lp: clr     dspmtx,x        ; clear first byte
        dbnzx   dcl1lp          ; clear 'em all
        clr     dspmtx          ; clear last one!
        rts                     ; all done

; -----------------------------------------------------------------------------
; Routine:      dclr2
; Called By:    main
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      dspmtx: The LED display matrix for the text display area
; Function:     This just clears line 2 of the display
; -----------------------------------------------------------------------------
dclr2:  ldx     #31             ; wanna clear 32 bytes
        clrh                    ; make sure H is zero
dcl2lp: clr     dspmtx+32,x     ; clear first byte
        dbnzx   dcl2lp          ; clear 'em all
        clr     dspmtx+32       ; clear last one!
        rts                     ; all done

        page
; -----------------------------------------------------------------------------
; Routine:      dton
; Called By:    main
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      dspmtx: The LED display matrix for the text display area
; Function:     simply writes all 1's to the display matrix to turn all LEDs on.
; -----------------------------------------------------------------------------
dton:   ldx     #63             ; wanna clear 64 bytes
        clrh                    ; make sure H is zero
        lda     #$ff            ; all 1's
dtonlp: sta     dspmtx,x        ; set first byte
        dbnzx   dtonlp          ; clear 'em all
        sta     dspmtx          ; set last one!
        rts                     ; all done

        page
; -----------------------------------------------------------------------------
; Routine:      dputc
; Called By:    main
; Calls:        None
; Gazintas:     dtx: X location for the left-most column of the character
;               dty: row # of top row (0-8 are legit)
;               A: Character to write
; Gazoutas:     dtx: incremented by 6 after done
; Globals:      dspmtx: The LED display matrix for the text display area
; Function:     This routine puts a character at an arbitrary place on the
;               screen.  The character is OR'd into the existing matrix
;               value. The character to put is a printable ASCII character or
;               the special characters (as they exist) coded into values $00-$1f
;               The characters are a 5x8 matrix in tables.asm
; -----------------------------------------------------------------------------
dputc:  sub     #32             ; (for now) no characters below ASCII space
        cmp     #127-32         ; make sure no values higher than 127 too
        bhi     dputcx          ; if so (or a negative value), exit
        clrx                    ; X:A will be 16-bit offset into table
        asla                    ; Multiply reg pair by 8 (x2)
        rolx                    ; x2
        asla                    ; x4
        rolx                    ; x4
        asla                    ; x8
        rolx                    ; x8
        add     #asctbl%256     ; add base address of the ascii table (LSB)
        sta     ctoff+1         ; save absolute address LSB
        txa                     ; get MSB of offset in A
        adc     #asctbl\256     ; add base address of the ascii table (MSB)
        sta     ctoff           ; save absolute address MSB

;       Start with the Y value.  As each row is four bytes, the byte offset is
;       simply the y value (range checked) times four.

        clr     xyoff           ; start with an offset of 0
        clr     xyoff+1         ; a 16-bit value
        lda     dty             ; get the Y value
        cmp     #8              ; only 0-8 are legit values
        bhi     dputcx          ; if illegit, just exit
        asla                    ; byte offset is Y value x 4 (x2)
        asla                    ; byte offset is Y value x 4 (x4)
        sta     xyoff+1         ; save offset

dpcdx:  lda     dtx             ; get X offset
        cmp     #25             ; largest legit value
        bhi     dputcx          ; if greater, exit without doing anything
        and     #$07            ; 3 LSBs become shifting offset
        sta     xshif           ; save that
        lda     dtx             ; get offset again
        lsra                    ; /2
        lsra                    ; /4
        lsra                    ; upper bits become byte offset
        add     xyoff+1         ; add Y part of offset (can't be >256)
        add     #dspmtx         ; add base of display matrix (can't be >256)
        sta     xyoff+1         ; save X and Y offset

;       At this point ctoff is the source address and xyoff is the destination.
;       Xyoff is the upper left byte that is going to be modified.
;       xshif is the number of bits to shift the value in the bit table to the
;       right from that location.  If the value is 0,1, or 2, then we only
;       have to modify one byte.  For values 3-7 we need to modify two sequential
;       bytes.  We then repeat the process at +4 byte intervals for each row of
;       the 5x8 matrix entry.

        mov     #8,rshif        ; row counter: want to copy eight rows

dpcrlp: mov     xshif,cshif     ; get a copy of the shift count
        ldhx    ctoff           ; get the source address
        lda     ,x              ; get the byte to write
        aix     #1              ; increment it
        sthx    ctoff           ; save updated address
        clrx                    ; X is now other shifted byte
        tst     cshif           ; see if we need to shift
        beq     dpcdsh          ; if so, done shifting
dpcslp: lsla                    ; left shift lower byte
        rolx                    ; rotate it into X
        dec     cshif           ; decrement shift count
        bne     dpcslp          ; loop till done

;       A is now the byte to OR into the reference byte and X is the byte to
;       OR into the next byte over.
dpcdsh: pshx                    ; save X for a bit
        ldhx    xyoff           ; get the address to merge to
        ora     ,x              ; merge bits
        sta     ,x              ; save it
        aix     #1              ; next byte
        pula                    ; get old X (MSB to merge) into A for merging
        ora     ,x              ; merge bits
        sta     ,x              ; save it
        aix     #3              ; +1+3 is now next row down
        sthx    xyoff           ; save it for next round

        dec     rshif           ; decrement row counter
        bne     dpcrlp          ; do all eight rows

dpcinc: lda     dtx             ; get X offset
        add     #6              ; +6 for next character
        sta     dtx             ; and save it

dputcx: rts                     ; all done.

        page
; -----------------------------------------------------------------------------
; Routine:      dputs
; Called By:    main
; Calls:        dputc
; Gazintas:     dtx: X location for the left-most column of the character
;               dty: row # of top row (0-8 are legit)
;               (putsbf): The string to write (5 chars max or null-terminated)
; Gazoutas:     None.
; Globals:      dspmtx: The LED display matrix for the text display area
; Function:     This routine draws a string at an arbitrary place on the
;               screen.  The string is OR'd into the existing matrix
;               value. The string is up to 5 bytes long or null-terminted if
;               shorter.
; -----------------------------------------------------------------------------
dputs:  mov     #putsbf,putspt  ; start off pointing at the first byte

dputsl: ldx     putspt          ; get byte address
        clrh                    ; make sure we have a good 16-bit address
        lda     ,x              ; get a byte
        beq     dputsx          ; if null, all done
        jsr     dputc           ; write it (dtx and dty pass through from caller)
        lda     putspt          ; get the pointer back
        cmp     #putsbf+4       ; was it the last byte
        beq     dputsx          ; if so, exit
        inca                    ; otherwise increment
        sta     putspt          ; and store it
        bra     dputsl          ; and loop

dputsx: rts                     ; all done

        page
; -----------------------------------------------------------------------------
; Routine:      dpmsg
; Called By:    main
; Calls:        dputs
; Gazintas:     HX: Pointer to message in Flash
; Gazoutas:     None.
; Globals:      dspmtx: The LED display matrix for the text display area
; Function:     This routine displays a message in flash to the screen.  It is
;               a shell around the dputs to make writing fixed messages easier.
;               Flash structure is:
;                 byte 1: dtx
;                 byte 2: dty
;                 bytes 3-7 up to 5 char message, null if <5
; -----------------------------------------------------------------------------
dpmsg:  mov     X+,dtx          ; copy dtx
        mov     X+,dty          ; copy dty
        mov     X+,putsbf       ; text byte 1
        mov     X+,putsbf+1     ; text byte 2
        mov     X+,putsbf+2     ; text byte 3
        mov     X+,putsbf+3     ; text byte 4
        mov     X+,putsbf+4     ; text byte 5
        bra     dputs           ; go to dputs & return from there

        page
; -----------------------------------------------------------------------------
; Routine:      dttim
; Called By:    timirq
; Calls:        b2as2, dputs
; Gazintas:     mins, hrs, day, month
;               blink = blink flag
; Gazoutas:     None
; Globals:      dspmtx: (through dputs)
; Function:     This is the routine that writes out the time and date to the
;               LED matrix display.  It is called by the timirq routine if
;               the variable newtim is set to non-zero.  This is set either
;               by the minutes incrementing by 1 or anything changing during
;               the time setting routine.
;               Blink flag is to determine if an element of the time is to
;               be blinking and bit 7 actually controls the blinking rate.
;               Hence a field is NOT written if its field bit is set AND the
;               bit 7 is set.  Since the only time we display the year is
;               if we are setting it, if blink[4] is set, we bypass the
;               time/date code and show the year instead.
;               blink[0] Minutes
;               blink[1] Hours
;               blink[2] Days
;               blink[3] Month
;               blink[4] Year
;               blink[5-6] Reserved
;               blink[7] blink off.
; -----------------------------------------------------------------------------
dttim:  clr     newtim          ; mark time as having been updated

        brclr   4,blink,dttnyr  ; If year bit not set in blink flag, show normal time
        jmp     dttyr           ; need a long jump to show year

dttnyr: ldhx    #$2020          ; two spaces
        sthx    putsbf          ; default to spaces for time row
        sthx    putsbf+3        ; default to spaces for time row

        ; Hours is a pain to convert to 12-hour format

        lda     blink           ; get blink flag
        and     #$82            ; if hours flag set and blink off mode
        cbeqa   #$82,dttmin     ; if blinking hours don't draw 'em

        lda     hrs             ; get the hours
        tst     rtchld          ; see if setting the time
        bne     hrsn0           ; if so, leave in 24-hour mode
hrs12:  cmp     #12             ; see if hours >12
        bls     hrsam           ; if 0-12, no processing (yet)
        sub     #12             ; otherwise subtract 12
hrsam:  tsta                    ; re-check A
        bne     hrsn0           ; if not 0 no special processing
        lda     #12             ; 00=12 midnight
hrsn0:  jsr     b2as2           ; convert hours to ASCII
        ldhx    atemp           ; get both bytes
        sthx    putsbf          ; save in right location

        ; Minutes is straight forward

dttmin: lda     blink           ; get blink flag
        and     #$81            ; if minutes flag set and blink off mode
        cbeqa   #$81,dttdr1     ; if blinking minutes don't draw 'em

        lda     mins            ; get minutes (no processing needed)
        jsr     b2as2           ; convert it
        ldhx    atemp           ; get both bytes
        sthx    putsbf+3        ; save in right location

dttdr1: mov     #':',putsbf+2   ; put the colon in
        mov     #0,dtx          ; left side
        mov     #0,dty          ; line 1

        ; if hours are 1-9, omit the leading 0 and shift the time over 3 pixels
        ; to center it.  But, only if not in time setting mode

        tst     rtchld          ; see if in time setting mode
        bne     shotim          ; if so, skip this stuff
        lda     putsbf          ; get hours 10's digit
        cmp     #'0'            ; see if a 0
        bne     shotim          ; if not 0, no shifting needed

        ldhx    putsbf+1        ; do a left-shift of the time string
        sthx    putsbf          ; 2 bytes done
        ldhx    putsbf+3        ;
        sthx    putsbf+2        ; 4 bytes done
        clr     putsbf+4        ; null terminate the string
        mov     #3,dtx          ; start 3 pixels over

shotim: jsr     dclr            ; clear the display buffer
        jsr     dputs           ; write it!

        ; now put in the date, pretty simple

        ldhx    #$2020          ; two spaces
        sthx    putsbf          ; default to spaces for date row
        sthx    putsbf+3        ; default to spaces for date row

        lda     blink           ; get blink flag
        and     #$88            ; if month flag set and blink off mode
        cbeqa   #$88,dttday     ; if blinking month don't draw 'em

        lda     month           ; get the month
        jsr     b2as2           ; convert it
        ldhx    atemp           ; get both bytes
        sthx    putsbf          ; save in right location

dttday: lda     blink           ; get blink flag
        and     #$84            ; if day flag set and blink off mode
        cbeqa   #$84,dttdr2     ; if blinking day don't draw 'em

        lda     day             ; get the day of month
        jsr     b2as2           ; convert it
        ldhx    atemp           ; get both bytes
        sthx    putsbf+3        ; save in right location

dttdr2: mov     #'/',putsbf+2   ; put the colon in
        mov     #0,dtx          ; left side
        mov     #8,dty          ; line 2
        jsr     dputs           ; write it!

dttex:  rts                     ; all done

;       We come here if the year bit is set in the blink flag.  Display the
;       year instead of the date and time.

dttyr:  jsr     dclr            ; easiest to clear the whole display first

        ldhx    #$3230          ; "20" as we don't blink first two digits of year
        sthx    putsbf          ; buffer to write
        ldhx    #$2020          ; two spaces as default for last two digits
        sthx    putsbf+2        ; buffer to write

        lda     blink           ; get blink flag (sets N)
        bmi     dttbyr          ; if blink off time, leave spaces.

        lda     year            ; get the year
        jsr     b2as2           ; convert it
        ldhx    atemp           ; get both bytes
        sthx    putsbf+2        ; save in right location

dttbyr: clr     putsbf+4        ; null-terminate string (<5 chars)
        mov     #3,dtx          ; start 3 pixels over
        mov     #0,dty          ; line 1
        jsr     dputs           ; write it!
        bra     dttex           ; return from common location

        page
; -----------------------------------------------------------------------------
; Routine:      dclr
; Called By:    main
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      dspmtx: The LED display matrix for the text display area
; Function:
; -----------------------------------------------------------------------------

; -----------------------------------------------------------------------------
; Routine:      dclr
; Called By:    main
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      dspmtx: The LED display matrix for the text display area
; Function:
; -----------------------------------------------------------------------------


