        page
******************************************************
** Copyright 2007 CMI Productions.                  **
** Lots of rights reserved.                         **
** Free for personal use as long as this copyright  **
** message is maintained in the source and the LCD  **
** copyright message is also preserved.  Commercial **
** use prohibited without CMIs written concent.     **
******************************************************
; -----------------------------------------------------------------------------
; File:         wwvb.asm
; Function:     WWVB acquisition/parsing code
; Routines:     wwvbsm: routine that gets called every 50ms
; -----------------------------------------------------------------------------
;
;       This file contains all the WWVB processing routines.  The basic idea
;       is a state machine that gets called every 50ms
;
;       Right now this code does the basic WWVB decode and time setting
;       to UTC (not UT1).  There are still a lot of things to do still
;       1! Check all the boundary conditions for time setting--there are a lot!
;          Leap seconds, DST transitions, other dates...
;       2! Set up the code to be called once a day, not just at power-up
;       3! User-settable time zone and DST observation
;       4) Error checking (by getting the time twice)
;       5) Calibrating the time (offset checking afer 24 hours) (may not implement)
;       6) Noise filtering of the WWVB signal
;       7) Confirm leap-seconds move to midnight UTC, not midnight local
;       8) take DST into account for leap seconds
;
        page
; -----------------------------------------------------------------------------
; Routine:      wwvbsm
; Called By:    timirq
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      wwvbst: The state machine state
; Function:     This is the main WWVB parsing routine.  It is called every 50ms
;               to see what to do.  wwvbst is the state variable that decides
;               what it should be doing.  Here are the states
;               0 : Idle (mainline code sets to 1 to fire things off)
;               1 : Initialize
;               2 : Wait for Marker 1
;               3 : Wait for Marker 2
;               4 : Get 60 bits of data (the time)
;               5 : Convert it to UTC
;               6 : Convert it to local time
;               7 : Wait for the marker to set the clock's time
;
; -----------------------------------------------------------------------------
wwvbsm: lda     wwvbst          ; Get the WWVB state
        beq     wwvbx           ; if 0, idle, exit fast

;       If the state machine is active, blink the upper right LED to show
;       the WWVB data

        lda     portd           ; get raw port value (WWVB data is in bit 5)
        nsa                     ; WWVB is now in bit 1
        lsra                    ; now in bit 0
        and     #$01            ; bit 0 only
        sta     wwvled          ; save it

;       If not zero, go index a table of routines to run based on the state

        lda     wwvbst          ; Get the WWVB state
        cmp     #7              ; see if greater than max possible state value
        bhi     wwver           ; if >max, then error, reset state machine

        asla                    ; state times two (16-bit entries)
        tax                     ; X for an index
        clrh                    ; H=0 for good indexing
        lda     wwvtbl,x        ; get MSB of routine
        ldx     wwvtbl+1,x      ; get LSB of routine
        psha                    ; push MSB on the stack
        pulh                    ; get it into H, HX is now routine address
        jmp     ,x              ; go to that routine, they jump back to wwvbx

wwvtbl: dw      wwvbx           ; 0 = idle (not used as we check earlier for speed)
        dw      wwvini          ; 1 = initialize variables for getting time
        dw      wwvwm           ; 2 = wait for the two Px markers
        dw      wwvwm           ; 3 = wait for the two Px markers
        dw      wwvgd           ; 4 = get data (pass 1)
        dw      wwvpt1          ; 5 = process the time (basic year, month, day, mins)
        dw      wwvpt2          ; 6 = process the time (offsets)
        dw      wwvst           ; 7 = Set the time (waiting for last marker)

wwver:  clr     wwvbst          ; got here if wwbst currupt, so default to idle
wwvbx:  rts                     ; After error (or normal exit), all done

        page
; -----------------------------------------------------------------------------
; Routine:      wwvini (state #1)
; Called By:    wwvbsm
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      wwvbt : Table cleared out in preparation for getting data
;               wwvbpt : Bit pointer cleared
;               wwvbst : incremented for next state
; Function:     This is where time reception starts.  Main line code sets the
;               wwvbst state variable to 0x01 which transitions the state machine
;               to this state.  From here on out the state machine takes over.
;               In this initialization state we set rtchld to stop the clock
;               and clear the bit buffer for receiving the WWVB data.
; -----------------------------------------------------------------------------
wwvini:
$IF WWVDBG
        mov     #5,wwvbst       ; the correct stat for our jump
        jmp     wwvdb           ; skip reading bits and implement a debug dataset
$ENDIF
        ldx     #7              ; wanna clear 7 bytes
        clrh                    ; H=0
wwvinl: clr     wwvbt,x         ; clear a byte in the bit buffer
        dbnzx   wwvinl          ; clear 7 bytes
        clr     wwvbt           ; clear 8th byte

        bclr    1,portc         ; enable the WWBV radio

        mov     #1,wwvbpt       ; clear our bit pointer (0 is marker, so 1 is first bit)
        clr     wwvstm          ; clear symbol time counter
        inc     wwvbst          ; on to next state after this one
        bra     wwvbx           ; all done initializing

        page
; -----------------------------------------------------------------------------
; Routine:      wwvwm (states #2 and #3)
; Called By:    wwvbsm
; Calls:        wwvsym
; Gazintas:     None
; Gazoutas:     None
; Globals:      None
; Function:     First thing to do in getting the time is to look for the pair
;               of markers that signify the start of a frame of data.  This
;               code is actually two states, incrementing if we see a marker,
;               and resetting if we see something else, if we get past two
;               markers in succession, then we're synchronized and go on to
;               getting data.
; -----------------------------------------------------------------------------
wwvwm:  jsr     wwvsym          ; see if we got a symbol
        beq     wwvwnx          ; If nothing, exit

        cmp     #wwvspm         ; if something, compare to position marker
        beq     wwvwmy          ; If yes, go increment state
        mov     #2,wwvbst       ; If something else (1 or 0 or gunk), start over
        bra     wwvwnx          ; and exit

wwvwmy: inc     wwvbst          ; if we got a Position marker, on to next state
wwvwnx: bra     wwvbx           ; and exit

        page
; -----------------------------------------------------------------------------
; Routine:      wwvgd (state #4)
; Called By:    wwvbsm
; Calls:        wwvsym
; Gazintas:     None
; Gazoutas:     None
; Globals:      None
; Function:     Once Synchronized, this is the state where we get our 60 bits
;               of data.  Here I just go get 60 bits of data, storing 1's and
;               0's.  Error checking is to ensure that marker bits are where
;               they are supposed to be and not where they are not supposed
;               to be.
; -----------------------------------------------------------------------------
wwvgd:  jsr     wwvsym          ; see if a symbol available
        beq     wwvgdx          ; if none, exit
        cbeqa   #wwvs1,wwvgdd   ; If a 1, do bit processing
        cbeqa   #wwvs0,wwvgdd   ; If a 0, do bit processing
        cbeqa   #wwvspm,wwvgdp  ; If a position marker, do marker processing
        bra     wwvgde          ; if anything else, error!

wwvgdp: ldx     wwvbpt          ; get our position pointer
        clrh                    ; correct indexing
        lda     wwvptb,x        ; get our position marker table entry
        beq     wwvgde          ; if not supposed to be a position marker, error
        lda     #wwvs0          ; otherwise just stuff in a 0
        bra     wwvgdg          ; go do it.

wwvgdd: psha                    ; save A for a bit
        ldx     wwvbpt          ; get our position pointer
        clrh                    ; correct indexing
        lda     wwvptb,x        ; get our position marker table entry
        pula                    ; get A back before deciding what to do
        bne     wwvgde          ; if not supposed to be data, error
                                ; otherwise fall into data saving code.

wwvgdg: rora                    ; put bit we want to save into the carry
        rol     wwvbt+7         ; rotate it into byte #8 (LSB)
        rol     wwvbt+6         ; and on to byte #7
        rol     wwvbt+5         ; and on to byte #6
        rol     wwvbt+4         ; and on to byte #5
        rol     wwvbt+3         ; and on to byte #4
        rol     wwvbt+2         ; and on to byte #3
        rol     wwvbt+1         ; and on to byte #2
        rol     wwvbt           ; and on to byte #1 (MSB)

        inc     wwvbpt          ; increment our bit counter
        lda     wwvbpt          ; get it
        cmp     #60             ; did we get the last one?
        bne     wwvgdx          ; if not, keep getting data
        inc     wwvbst          ; otherwise increment to next state!
        bra     wwvgdx          ; and exit

wwvgde: mov     #1,wwvbst       ; if we got bad data, need to start over

wwvgdx: bra     wwvbx           ; all done

        page
; -----------------------------------------------------------------------------
; Routine:      wwvpt1 (state #5)
; Called By:    wwvbsm
; Calls:        cnvbcd
; Gazintas:     None
; Gazoutas:     None
; Globals:      None
; Function:     This is the function that takes the WWVB data and converts it
;               it to local format.  I parse the time and write it to the
;               holding bytes before writing it to the official time.
;               Since WWVB time is BCD and I'm binary, there is quite a bit
;               of converting going on!  Note that I have a 64-bit buffer, but
;               only 59 of the 60 bits get shifted in I don't bother shifting in
;               the initial Pr bit.  This means that my array is offset somewhat
;               from the WWVB table.  The Minutes 40-bit is in byte 0, bit 2.
;               I May have to split this into a couple of time slices if the
;               conversions take too long.
;                10s-Minutes = byte0[2:0]          (3 bits)
;                 1s-Minutes = byte1[6:3]          (4 bits)
;                10s-Hours   = byte2[7:6]          (2 bits)
;                 1s-Hours   = byte2[4:1]          (4 bits)
;               100s-DOY     = byte3[5:4]          (2 bits)
;                10s-DOY     = byte3[2:0]byte4[7]  (4 bits)
;                 1s-DOY     = byte4[5:2]          (4 bits)
;                10s-Year    = byte6[6:3]          (4 bits)
;                 1s-Year    = byte6[1:0]byte7[7:6 (4 bits)
;               The various marker bits (leap year, leap second, and DST) don't
;               need parsing.  This just decodes the time as sent by WWVB,
;               the next state converts it to localt time.
; -----------------------------------------------------------------------------
; This is some test code to copy some test bytes to the table and make sure
; the code below decodes it correctly.  This snippet is debug code only, the
; proper routine starts at wwvpt1.  See the table of test sequences in
; tables.asm.
$IF WWVDBG
wwvdb:  ldhx    #wwvdbg         ; the start of the test sequence
        lda     ,x              ; get first byte
        sta     wwvbt           ; save it
        lda     1,x             ; get next byte
        sta     wwvbt+1         ; save it
        lda     2,x             ; get next byte
        sta     wwvbt+2         ; save it
        lda     3,x             ; get next byte
        sta     wwvbt+3         ; save it
        lda     4,x             ; get next byte
        sta     wwvbt+4         ; save it
        lda     5,x             ; get next byte
        sta     wwvbt+5         ; save it
        lda     6,x             ; get next byte
        sta     wwvbt+6         ; save it
        lda     7,x             ; get next byte
        sta     wwvbt+7         ; save it
$ENDIF

wwvpt1: lda     wwvbt+0         ; byte 0 has minutes 10's bits (bits 2:0)
        nsa                     ; put them in bits 6:4
        sta     wwmins          ; save temporarily
        lda     wwvbt+1         ; byte 1 has minutes 1's bits (bits 6:3)
        rola                    ; bits now 7:4
        nsa                     ; bits now 3:0
        ora     wwmins          ; merge in 10's bits
        and     #$7f            ; A is now minutes in BCD
        jsr     cnvbcd          ; convert to binary
        sta     wwmins          ; ready for programming

        lda     wwvbt+2         ; byte 2 has hours 10's (bits 7:6)
        rora                    ; now bits 6:5
        rora                    ; now bits 5:4
        and     #$30            ; only these bits
        sta     wwhrs           ; save it
        lda     wwvbt+2         ; byte 2 has hours 1's (bits 4:1)
        rora                    ; now bits 3:1
        and     #$0f            ; only these bits
        ora     wwhrs           ; merge in 10's bits
        jsr     cnvbcd          ; convert to binary
        sta     wwhrs           ; and save it

        lda     wwvbt+4         ; byte 4 has DOY 10's (bit 7)
        rola                    ; put that bit in the carry
        lda     wwvbt+3         ; byte 3 has DOY 10's (bits 2:0)
        rola                    ; A now has DOY 10's (bits 3:0)
        nsa                     ; now in bits 7:4
        and     #$f0            ; only those bits
        sta     wwday           ; save it for a bit
        lda     wwvbt+4         ; byte 4 has DOY 1's (bits 5:2)
        rora                    ; now its 4:1
        rora                    ; now bits 3:0
        and     #$0f            ; only those bits
        ora     wwday           ; merge bits
        jsr     cnvbcd          ; convert to binary (only 0-99 so far)
        brclr   4,wwvbt+3,wwvpdc; see if DOY 100 bit set
        add     #100            ; add a hundred
wwvpdc: deca                    ; my original math assumed 1/1/xx was day 0
;                               ; it is day #1, so quick cheesy fix
        sta     wwday           ; save result except for 200 bit

; Processing the 200's bit is a pain as I want to avoid 16-bit math.  The
; solution is that if the 200's bit is set, I preload the month with July 1
; (day 181/182 depending on leap year) and increment the days count by 19.
; That way, as I subtract months below, I essentially start off with the
; 200th day and keep subtracting using only 8-bit math.

        clr     wwmonth         ; default to January first
        brclr   5,wwvbt+3,wwvpdl; if 200-bit isn't set, normal processing
        mov     #6,wwmonth      ; otherwise start with July
        lda     wwday           ; get the day count
        add     #19             ; add 18 to it (June 1 is day 181 if no leap year)
        sta     wwday           ; save it
        brclr   4,wwvbt+7,wwvpdl; if no leap year, done
        dec     wwday           ; if leap year, decrement one day

wwvpdl: lda     wwmonth         ; get the month (0-11)
        cmp     #1              ; see if february (1 in 0-based month number)
        bne     wwvpdf          ; if not, normal processing
        brclr   4,wwvbt+7,wwvpdf; if not a leap year, "normal" Feb days
        lda     #29             ; if Feb and leap year, 29 days.
        bra     wwvpdd          ; and process normally

wwvpdf: ldx     wwmonth         ; get month (0-11)
        clrh                    ; good index
        lda     mthtbl,x        ; get days in month

wwvpdd: cmp     wwday           ; compare to days in month to left in year
        bhi     wwvpdy          ; if month has more days to leftover, done!
        sta     atemp           ; Otherwise not done, save days in current month
        lda     wwday           ; get days left
        sub     atemp           ; subtract days in current month (will be >1)
        sta     wwday           ; save leftover days
        inc     wwmonth         ; and increment the month
        bra     wwvpdl          ; and go get next month

wwvpdy: lda     wwvbt+6         ; byte 6 has Years 10's in bits (bits 6:3)
        rola                    ; bits now 7:4
        and     #$f0            ; only those bits
        sta     wwyear          ; save 'em
        lda     wwvbt+6         ; byte 6 has years 1's bits 3:2 in bits 1:0
        ldx     wwvbt+7         ; byte 7 has years 1's bits 1:0 in bits 7:6
        rolx                    ; need to shift X into A two bits
        rola                    ; ...
        rolx                    ; ...
        rola                    ; now A[3:0] has years 1's bits!
        and     #$0f            ; only those bits
        ora     wwyear          ; merge MSD and LSD
        jsr     cnvbcd          ; convert to binary
        sta     wwyear          ; save result

        inc     wwday           ; make days 1-31 instead of 0-30
        inc     wwmonth         ; make month 1-12 instead of 0-11

        inc     wwvbst          ; enough for this interrupt, wait for next state

        clr     wwvlsf          ; clear the leap second flag
        brclr   3,wwvbt+7,wwvptx; if WWVB table bit not set, don't set my flag
        inc     wwvlsf          ; otherwise set the flag

wwvptx: jmp     wwvbx           ; all done

        page
; -----------------------------------------------------------------------------
; Routine:      wwvpt2 (state #6)
; Called By:    wwvbsm
; Calls:        mthmax
; Gazintas:     wwmins,wwhrs,wwday,wwmonth,wwyear
; Gazoutas:     wwmins,wwhrs,wwday,wwmonth,wwyear
; Globals:      None
; Function:     This function takes care of the offsets to the time just
;               decoded.  This includes the offset to the local time from UCT,
;               Daylight Savings Time, adding a minute (as the broadcast time
;               is one minute after the actual minute), and (eventually)
;               the sub-second UT1 offset if I determine that matters.
; -----------------------------------------------------------------------------
wwvpt2: inc     wwmins          ; increment minutes
        lda     wwmins          ; get incremented value
        cmp     #60             ; see if 60 minutes
        blo     wwv2hr          ; if not, done with minutes
        clr     wwmins          ; otherwise clear minutes and...
        inc     wwhrs           ; increment hours

wwv2hr: clr     wwvdsf          ; clear the DST transition flag
        lda     fidso           ; check the DST observation flag
        beq     wwv2st          ; if false (no DST observation), go to ST

        brclr   2,wwvbt+7,wwv261; start splitting DS bit combinations
        brclr   1,wwvbt+7,wwv2td; [57:58]=10 means transitioning to DST today
        bra     wwv2ds          ; [57:58]=11 means full DST
wwv261: brclr   1,wwvbt+7,wwv2st; [57:58]=00 means full ST, no processing needed

;       We get here if this is the day we're transitioning to Standard Time
;       (fall back). If the standard time is <1 (midnight-1am), treat as DST but
;       set the DST flag to fall back at 2am local
wwv2ts: lda     wwhrs           ; get the UTC hours
        ldx     fitz            ; get timezone index
        clrh                    ; h=0...
        add     tztbl,x         ; add local offset (to ST)
        cmp     #1              ; compare to 1am ST
        bhi     wwv2st          ; after 1am (ST), we're at ST
        mov     #1,wwvdsf       ; otherwise flag to convert to ST at 2am local
        bra     wwv2ds          ; and stay in DST mode

;       We get here if this is the day we;re transitioning to Daylight Savings
;       Time (Spring forward).  If the standard time is after 2am, treat as
;       DST otherwise assume ST and set the flag to spring forward at 2am.
wwv2td: lda     wwhrs           ; get the UTC hours
        ldx     fitz            ; get timezone index
        clrh                    ; h=0...
        add     tztbl,x         ; add local offset (to ST)
        cmp     #2              ; compare to 2am ST
        bhi     wwv2ds          ; if after 2am, assume DST in effect
        mov     #2,wwvdsf       ; otherwise flag to convert to DST at 2am local
        bra     wwv2st          ; and stay with standard time

wwv2ds: inc     wwhrs           ; if DST in effect, increment hours now
wwv2st: lda     wwhrs           ; get hours (adjusted)
        ldx     fitz            ; get timezone index
        clrh                    ; h=0...
        add     tztbl,x         ; add local offset (to ST)
        sta     wwhrs           ; just in case done, store hours
        bmi     wwv2hm          ; if negative, need to decrement a day
        cmp     #24             ; see if >24 hours
        blo     wwv2x           ; if not (and not negative), done!

        sub     #24             ; if >24, subtract 24 hours since we're in to the next day
        sta     wwhrs           ; and save hours

;       Now is a copy of the code to increment a day...

        inc     wwday           ; increment day-of-month
        lda     wwmonth         ; get month
        ldx     wwyear          ; get year
        jsr     mthmax          ; get month max value in A

        cmp     wwday           ; compare to current day count
        bhs     wwv2x           ; if days-in-month higher, then no action
        lda     #1              ; new day is 1st of month
        sta     wwday           ; remember that

        inc     wwmonth         ; increment month
        lda     wwmonth         ; get incremented value
        cmp     #13             ; see if December
        blo     wwv2x           ; if not, done
        lda     #1              ; new month is January
        sta     wwmonth         ; remember that

        inc     wwyear          ; increment year
        lda     wwyear          ; get incremented value
        cmp     #100            ; see if 100 year (Y2K!)
        blo     wwv2x           ; if not, done
        clr     wwyear          ; otherwise clear year to 00
        bra     wwv2x           ; all done

;       This is the code to decrement an hour, taking care of day/month/year boundary conditions

wwv2hm: lda     wwhrs           ; get the hours back
        add     #24             ; went to previous day, so add 24 hours
        sta     wwhrs           ; save corected hours (no error possible, know we were <24)

        dec     wwday           ; decrement day-of-month
        lda     wwday           ; get it
        bne     wwv2x           ; if not 0, done!

        dec     wwmonth         ; decrement month
        lda     wwmonth         ; get decremented value
        bne     wwv2gm          ; if not 0, done, get day-of-month
        mov     #12,wwmonth     ; new month is December
wwv2gm: lda     wwmonth         ; get month
        ldx     wwyear          ; get year
        jsr     mthmax          ; get month max value in A
        sta     wwday           ; remember that
        lda     wwmonth         ; get the month back
        cmp     #12             ; see if dec (remembering we rolled back one year)
        bne     wwv2x           ; if not, don't decrement the year

        dec     wwyear          ; decrement year
        bpl     wwv2x           ; if not -1, then OK
        mov     #99,wwyear      ; otherwise clear year to 99

wwv2x:  inc     wwvbst          ; on to the final state to start the time
        jmp     wwvbx           ; all done

        page
; -----------------------------------------------------------------------------
; Routine:      wwvst (state #7)
; Called By:    wwvbsm
; Calls:        None
; Gazintas:     wwmins,wwhrs,wwday,wwmonth,wwyear
; Gazoutas:     ssecs,secs,hrs,day,month,year
; Globals:      None
; Function:     This is the routine to actually set the real time to the WWVB
;               time.  We've already gotten the first of the two marker bits,
;               so are just waiting for the leading edge of the second one to
;               synchronize to.  This routine gets called every 50ms so at
;               least for now, that is my accuracy limit.
; -----------------------------------------------------------------------------
wwvst:  brclr   5,portd,wwvstx  ; check the WWVB signal, if not xmitting, do nuthin

;       Otherwise leading edge!  Set the time!

        mov     #1,rtchld       ; stop real time (not really needed, we're in an IRQ)
        clr     secs            ; set seconds to zero
        clr     ssecs           ; set the sub-seconds to zero
        lda     wwmins          ; get wwvb minutes
        sta     mins            ; make 'em real time
        lda     wwhrs           ; get wwvb hours
        sta     hrs             ; make 'em real time
        lda     wwday           ; get wwvb day of month
        sta     day             ; make it real time
        lda     wwmonth         ; get wwvb month
        sta     month           ; make it real time
        lda     wwyear          ; get the wwvb years
        sta     year            ; make it real time

;       Make sure the WWVB led is turned off
        bset    0,clkmtx+8      ; turn the tick mark for noon back on
        bset    0,clkmty+8      ; turn the tick mark for noon back on

        clr     rtchld          ; start counting time
        mov     #1,newtim       ; flag a new time (for LED matrix update)

        bset    1,portc         ; disable the WWBV radio

        clr     wwvbst          ; all done with wwvb state machine!
wwvstx: jmp     wwvbx           ; all done with routine

        page
; -----------------------------------------------------------------------------
; Routine:      wwvsym
; Called By:    wwvwm, wwvgd
; Calls:        None
; Gazintas:     None
; Gazoutas:     A = Symbol detected (if any)
; Globals:      None
; Function:     This routine looks for a symbol.  It does so by getting called
;               every 50ms and counting high states from the WWVB radio.  On
;               the first low state it looks at how many high states there
;               were before and based on that determines what the symbol was.
;               0 bit = 200ms = 3-5 high samples   = returns wwvs0
;               1 bit = 500ms = 9-11 high samples  = returns wwvs1
;               P bit = 800ms = 15-17 high samples = returns wwvspm
;               anything else is gunk              = returns wwvsxx
;               Useful data is only returned on first low sample.  If it isn't
;               a first high low, this routine returns wwvsnn
;
;               Not yet, but I do need to add some basic filtering sometime.
; -----------------------------------------------------------------------------
wwvsym: brclr   5,portd,wwvsy0  ; check the WWVB signal (1=transmitting, 0=not)

wwvsy1: inc     wwvstm          ; if asserted (bit time), increment bit-time counter
        bra     wwvsns          ; and done. No over checking needed, return no symbol yet

wwvsy0: lda     wwvstm          ; see if our first low value (after bit time)
        beq     wwvsns          ; if wwvstm=0, not our first unasserted value, just exit "no symbol"
        cmp     #17             ; see if >17 asserted states
        bhi     wwvsmg          ; if so, it's gunk
        tax                     ; otherwise start making an index
        clrh                    ; H=0
        lda     wwvsyt,x        ; get symbol from table
        clr     wwvstm          ; reset low state counter to 0
        bra     wwvsmx          ; and exit

wwvsns: lda     #wwvsnn         ; Nothing yet symbol
        bra     wwvsmx          ; and exit

wwvsmg: lda     #wwvsxx         ; gunk symbol
        clr     wwvstm          ; reset low state counter to 0

wwvsmx: sta     ramff           ; for debug
        tsta                    ; set flag bits for A
        rts                     ; all done

        page
; -----------------------------------------------------------------------------
; Routine:
; Called By:    wwvbsm
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      None
; Function:
; -----------------------------------------------------------------------------

