        page
******************************************************
** Copyright 2007 CMI Productions.                  **
** Lots of rights reserved.                         **
** Free for personal use as long as this copyright  **
** message is maintained in the source and the LCD  **
** copyright message is also preserved.  Commercial **
** use prohibited without CMIs written concent.     **
******************************************************
; -----------------------------------------------------------------------------
; File:         dtime.asm
; Function:     Routines for displaying the time on the outer rings
; Routines:     tclr: clear out the time display (all LEDs off)
;               dtime: show the time (for now that's it)
; Globals:      ssecs, secs, mins, hrs: The time
;               clkmtx: The LED display matrix for the three rings of LEDs
;               ssmode: Sub-second display mode
;                 0: No display of sub-seconds.
;                 1: Single LED whips around like seconds except all 60 in 1 second.
;                 2: As above except seconds are now a solid bar
;                 3: Two LEDs each going opposite directions.
;                 4: Subseconds go clockwise on even seconds, counter-clockwise on odd.
; -----------------------------------------------------------------------------

;       The clock matrix rendering routines
;       This file is responsible for the routines that fill out the clkmtx array
;       with the right data to display the time in the three rings of LEDs.
;       The details of that array structure are in the file "LED Matrix.txt".
;       These routines do not do the actual display refresh, that is done in
;       timirq.asm.
;
        page
; -----------------------------------------------------------------------------
; Routine:      tclr
; Called By:    irqrtn
; Calls:        None
; Gazintas:     None
; Gazoutas:     None
; Globals:      clkmtx: The LED display matrix for the three rings of LEDs (Phase A)
;               clkmty: Phase B
; Function:     Clears out the display matrix to all 0's (all LEDs off)
; -----------------------------------------------------------------------------
tclr:   ldx     #31             ; wanna clear 32 bytes
tclrlp: clr     clkmtx,x        ; clear first byte
        dbnzx   tclrlp          ; clear 'em all
        clr     clkmtx          ; clear last one!
        rts                     ; all done

        page
; -----------------------------------------------------------------------------
; Routine:      dtime
; Called By:    main
; Calls:        some smaller support routines
; Gazintas:     None
; Gazoutas:     None
; Globals:      ssecs, secs, mins, hrs: The time
;               ssmode: Sub-seconds display mode
;               clkmtx: The LED display matrix for the three rings of LEDs
;               rtchld: If time isn't incrementing, don't show seconds or sub-secs.
; Function:     Renders the time to clkmtx.  Display routines for hours, minutes,
;               and seconds are all the same, ssmode decides how (or if) to
;               render the sub-seconds.
; -----------------------------------------------------------------------------
;       First clear out the minutes/seconds array and pre-fill the hours array
;       with just the tick marks.

dtime:  ldx     #7              ; wanna clear 8 bytes (minutes/seconds bytes)
        clrh                    ; make sure HX is just X
clkmt2: clr     clkmtx,x        ; clear first byte (Phase A)
        clr     clkmty,x        ; Clear Phase B
        dbnzx   clkmt2          ; clear 'em all
        clr     clkmtx          ; clear last one! (Phase A)
        clr     clkmty          ; clear last one! (Phase B)

        ldx     #7              ; wanna copy 8 bytes (hours bytes)
        clrh                    ; make sure HX is just X
clkmt3: lda     tckmtx,x        ; get a byte of the tick-only array
        sta     clkmtx+8,x      ; store in hours part of array (Phase A)
        sta     clkmty+8,x      ; Store Phase B too
        dbnzx   clkmt3          ; do 15 of the 16 bytes

;       For the first byte we get the byte from the table, and if we are in
;       time getting mode, OR in the WWV LED value

        lda     tckmtx          ; get last byte
        tst     wwvbst          ; see if wwvb state machine is idle
        beq     clkmt3a         ; if so, just store it
        and     #$fe            ; kill bit 0
        ora     wwvled          ; add the WWV LED state

clkmt3a:sta     clkmtx+8        ; store last byte (Phase A)
        sta     clkmty+8        ; store last byte (Phase B)

;       Fill in the hours.  There is an LED per quarter hour, so I first
;       get a 0-3 index from the minutes value then add 4x the hour count to
;       get a 0-47 value.  I then have a table that tells me what byte/bit to
;       turn on based on that value.  I OR that value into the hours array.

dohrs:
        lda     hrs             ; get hours (0-23)
        cmp     #12             ; see if >=12
        blo     dohrnm          ; if not normal processing
        sub     #12             ; otherwise subtract 12
dohrnm: asla                    ; hours times two
        asla                    ; hours times four (table is four entries per hour
        ldx     mins            ; get minutes as an indes (to get quarter-hour offset)
        clrh                    ; HX is index
        add     qhrtbl,x        ; get 0-3 offset based on minutes

        tax                     ; in X (H still 0)
        aslx                    ; times two
        lda     hrstbl+1,x      ; get value to write
        ldx     hrstbl,x        ; get offset
        ora     clkmtx+8,x      ; merge in existing matrix value
        sta     clkmtx+8,x      ; write it back. (Phase A only, so blinks)

;       Next is minutes.  For each minutes value I have a table that has the
;       byte offset and bit to set.  Note that for minutes (only) I do not
;       OR in the value, the array is already zeros and this is the first
;       value to write.  All subsequent adds to the array must be ORd in.

domins: ldx     mins            ; get minutes
        aslx                    ; times two (two bytes per entry)
        clrh                    ; make it an index
        lda     mintbl+1,x      ; get value to write
        ldx     mintbl,x        ; get offset
        sta     clkmtx,x        ; write it back. (Phase A only, so it blinks)

;       Next is seconds.  For each seconds value I use the same minutes table
;       to find out which byte offset and bit value to add to the LED refresh
;       matrix.  From here on out the bit to turn on must be ORd with any
;       existing values.  If time is on hold, it is because we are setting the
;       time, so if rtchld is non-zero, don't show seconds or subseconds.

dosecs: tst     rtchld          ; check RTC hold flag
        bne     ssmdun          ; if not 0, exit now
        lda     ssmode          ; get mode
        cmp     #2              ; see if mode #2 (don't do secs in this mode)
        beq     ssdmod          ; if so, go on to mode displays
        ldx     secs            ; get seconds
        aslx                    ; times two (two bytes per entry)
        clrh                    ; make it an index
        lda     mintbl+1,x      ; get value to write
        psha                    ; save it
        ldx     mintbl,x        ; get offset
        ora     clkmtx,x        ; merge in existing matrix value
        sta     clkmtx,x        ; write the LED value
        pula                    ; byte to merge back
        ora     clkmty,x        ; merge in existing matrix value (Phase B)
        sta     clkmty,x        ; write the LED value (Phase B)

;       That is the "normal stuff"  For 1/60ths of a second, I get to play.
;       ssmode is a variable that lets me pick from various creative ways of
;       showing 1/60ths of a second.  See file header for modes
;       For a leap-second though, override everything!

ssdmod: lda     secs            ; get seconds back again
        cmp     #60             ; see if a leap second
        bne     ssddmd          ; if not, normal sub-second processing

;       For a leap second, flash the outer ring!
        ldx     #15             ; wanna set 16 bytes
        lda     #$55            ; alternating bits
tlpslp: sta     clkmtx,x        ; clear first byte
        coma                    ; alternating patter
        sta     clkmty,x        ; save that
        coma                    ; and back to non-negated
        dbnzx   tlpslp          ; clear 'em all
        sta     clkmtx          ; clear last one!
        coma                    ; and back to non-negated
        sta     clkmty          ; save that

ssddmd: lda     ssmode          ; get the sub-second mode
        beq     ssmod0          ; if 0 mode 0
        deca                    ; mode 1
        beq     ssmod1          ; go do mode 1
        deca                    ; mode 2
        beq     ssmod2          ; go do mode 2
        deca                    ; mode 3
        beq     ssmod3          ; go do mode 3
        deca                    ; mode 4
        beq     ssmod4          ; go do mode 4
                                ; otherwise fall through to mode 0

ssmod0: bra     ssmdun          ; do nuthin'

ssmod1: jsr     ssdn            ; do the normal 1/60ths display
        bra     ssmdun          ; all done

ssmod2: jsr     sssbar          ; do seconds bar
        bra     ssmdun          ; all done

ssmod3: jsr     ssdn            ; do the normal 1/60ths display
        jsr     ssdb            ; do it backwards too
        bra     ssmdun          ; all done

ssmod4: lda     secs            ; get seconds
        and     #$01            ; LSB only
        bne     ssmd4o          ; if odd, do odd processing
        jsr     ssdn            ; if even do normal sequence
        bra     ssmdun          ; and exit
ssmd4o: jsr     ssdb            ; if odd, so backwards sequence
        bra     ssmdun          ; and exit

ssmdun: rts                     ; done setting up time

        page
; -----------------------------------------------------------------------------
;       Normal sequencing, single LED going clockwise in 1/60ths of a second
; -----------------------------------------------------------------------------

ssdn:   ldx     ssecs           ; get sub-seconds
        aslx                    ; times two
        clrh                    ; make it an index
        lda     mintbl+1,x      ; get value to write
        psha                    ; save it
        ldx     mintbl,x        ; get offset
        ora     clkmtx,x        ; merge in existing matrix value
        sta     clkmtx,x        ; write it back.
        pula                    ; byte to merge back
        ora     clkmty,x        ; merge in existing matrix value (Phase B)
        sta     clkmty,x        ; write the LED value (Phase B)
        rts                     ; all done

; -----------------------------------------------------------------------------
;       Backwards.  LEDs go counter-clockwise in 1/60ths of a second
; -----------------------------------------------------------------------------

ssdb:   lda     #59             ; 59...
        sub     ssecs           ; get sub-seconds inverted
        tax                     ; make an index
        aslx                    ; times two
        clrh                    ; make it an index
        lda     mintbl+1,x      ; get value to write
        psha                    ; save it
        ldx     mintbl,x        ; get offset
        ora     clkmtx,x        ; merge in existing matrix value
        sta     clkmtx,x        ; write it back.
        pula                    ; byte to merge back
        ora     clkmty,x        ; merge in existing matrix value (Phase B)
        sta     clkmty,x        ; write the LED value (Phase B)
        rts                     ; all done

; -----------------------------------------------------------------------------
;       Seconds bar: fill in 0-seconds as a solid bar and then invert the
;       sub-seconds value.
; -----------------------------------------------------------------------------

sssbar: ldhx    #clkmtx         ; base of the clock matrix
        lda     secs            ; get seconds
        inca                    ; need to go 1-60, not 0-59 for this bar
ssbalp: cmp     #8              ; see if <8
        blo     ssba1           ; if so, done with bytes, do bits
        psha                    ; save count
        lda     #$ff            ; all eight LEDs on
        eor     ,x              ; make it so (Phase A)
        sta     ,x              ; and write it back
        lda     #$ff            ; all eight LEDs on
        eor     16,x            ; make it so (Phase B)
        sta     16,x            ; and write it back
        aix     #1              ; on to next byte
        pula                    ; get count back
        sub     #8              ; decrement seconds countby 8 bytes
        bra     ssbalp          ; and do next eight

ssba1:  pshh                    ; save H
        pshx                    ; save X
        clrh                    ; 0...
        tax                     ; HX is now index into bartbl
        lda     bartbl,x        ; get it
        pulx                    ; get X back
        pulh                    ; get H back
        psha                    ; save A now!
        eor     ,x              ; merge value (Phase A)
        sta     ,x              ; and write it back
        pula                    ; get merge value back
        eor     16,x            ; merge value (Phase B)
        sta     16,x            ; and write it back

        rts                     ; all done


