#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <math.h>
#include "def.h"
#include "Simulation.h"
#include "params.h"
#include "sorts.h"
#include "reports.h"
#include "util.h"

/****************************************************************************
* Routines : report_percentile
* Gazintas : results (the array of individual simulation results
* IOs      : None
* Returns  : None
* Globals  : params
*
* This is the routine to report back the results of the simulation sorted
* by percentile.
****************************************************************************/
void report_percentile(RESULTS* results, FILE* fout)
{
   uint32 i,j  ; /* general counters   */
   char amt1[16] ;
   char amt2[16] ;
   char amt3[16] ;

   /* now sort the results by normal mode estate  */
   qsort(results,params.num_sims,sizeof(RESULTS),(void*)res0sort) ;

   /* if requested, print out the sorted results--can be a lot! */
   if (params.detail_summary)
   {
      fprintf (fout,"Raw EOL Results for all simulations (Sorted by Mode 0 results):\n") ;
      fprintf (fout," avg_ror       estate0       estate1       estate2\n") ;
      fprintf (fout,"======== ============= ============= =============\n") ;
      for (i=0;i<params.num_sims;i++)
      {
         itoc(amt1,(int32)results[i].estate[0],10,'$',TRUE) ;
         itoc(amt2,(int32)results[i].estate[1],10,'$',TRUE) ;
         itoc(amt3,(int32)results[i].estate[2],10,'$',TRUE) ;
         fprintf (fout,"%8.4f %13s %13s %13s\n",results[i].avg_ror,amt1,amt2,amt3) ;
      }
      fprintf (fout,"\n") ;
   } ;

   /* print out the percentiles for each mode  */
   fprintf (fout,"\nPercentiles (normal):\n") ;
   fprintf (fout,"percentile       RoR       estate0\n") ;
   fprintf (fout,"========== ========= =============\n") ;
   for (i=1;i<10;i++)
   {
     j=i*params.num_sims/10 ;
     itoc(amt1,(int32)results[j].estate[0],10,'$',TRUE) ;
     fprintf (fout,"       %2d%% %8.4f%% %13s\n",i*10,results[j].avg_ror,amt1) ;
   }
   fprintf (fout,"\n") ;

   /* now sort the results by integrated mode estate  */
   qsort(results,params.num_sims,sizeof(RESULTS),(void*)res1sort) ;

   /* print out the percentiles */
   fprintf (fout,"\nPercentiles (integrated):\n") ;
   fprintf (fout,"percentile       RoR       estate1\n") ;
   fprintf (fout,"========== ========= =============\n") ;
   for (i=1;i<10;i++)
   {
     j=i*params.num_sims/10 ;
     itoc(amt1,(int32)results[j].estate[1],10,'$',TRUE) ;
     fprintf (fout,"       %2d%% %8.4f%% %13s\n",i*10,results[j].avg_ror,amt1) ;
   }
   fprintf (fout,"\n") ;

      /* now sort the results by integrated mode estate  */
   qsort(results,params.num_sims,sizeof(RESULTS),(void*)res2sort) ;

   /* print out the percentiles */
   fprintf (fout,"\nPercentiles (cache):\n") ;
   fprintf (fout,"percentile       RoR       estate2\n") ;
   fprintf (fout,"========== ========= =============\n") ;
   for (i=1;i<10;i++)
   {
     j=i*params.num_sims/10 ;
     itoc(amt1,(int32)results[j].estate[2],10,'$',TRUE) ;
     fprintf (fout,"       %2d%% %8.4f%% %13s\n",i*10,results[j].avg_ror,amt1) ;
   }
   fprintf (fout,"\n") ;

} ;

/****************************************************************************
* Routines : report_histogram
* Gazintas : results (the array of individual simulation results
* IOs      : None
* Returns  : None
* Globals  : params
*
* This is the routine to report back the results of the simulation in a
* 100-point histogram.
****************************************************************************/
void report_histogram (SUMMARY* summaries,RESULTS* results, FILE* fout, char* csvfname)
{
   uint32 i,j                     ; /* general counters   */
   uint32 bin                     ; /* histogram bin #    */
   float hmin, hmax               ; /* histogram min/max  */
   uint32 histogram[N_MODES][100] ; /* Results Histograms */
   FILE* fcsv                     ; /* CSV file           */
   char amt1[16] ;
   char amt2[16] ;

   /* only do this report if requested */
   if (params.histogram==FALSE) return ;

   /* Create a 100-point histogram of the results */
   /* The 100 bins are evenly spaced between the min and max results. */
   /* The histogram is for mode 1 and mode 2 data */

   /* first clear out the histogram array */
   for (i=0;i<100;i++)
   {
      for (j=0;j<N_MODES;j++)
      {
         histogram[j][i]=0 ;
      }
   }

   /* determine the min and max for all modes */
   /* start with first, then compare to rest  */
   hmin=summaries[0].min_left ;
   hmax=summaries[0].max_left ;
   for (j=1;j<N_MODES;j++)
   {
      if (summaries[j].min_left<hmin) hmin=summaries[j].min_left ;
      if (summaries[j].max_left>hmax) hmax=summaries[j].max_left ;
   } ;

   /* now round things to the nearest thousand */
   hmin=floor(hmin/1000.0)*1000.0 ;
   hmax=ceil(hmax/1000.0)*1000.0 ;

   /* fill in the histogram */
   for (i=0;i<params.num_sims;i++)
   {
      for (j=0;j<N_MODES;j++)
      {
         bin=(results[i].estate[j]-hmin)/(hmax-hmin)*100.0 ;
         histogram[j][bin]++ ;
      }
   }

   /* print out the histogram values */
   fprintf (fout,"\n100-point histogram spread\n") ;
   fprintf (fout,"         from              to mode0 mode1 mode2\n") ;
   fprintf (fout,"=============   ============= ===== ===== =====\n") ;
   for (i=0;i<100;i++)
   {
      itoc(amt1,(int32)(hmin+(hmax-hmin)*  i  /100),10,'$',TRUE) ;
      itoc(amt2,(int32)(hmin+(hmax-hmin)*(i+1)/100),10,'$',TRUE) ;
      fprintf (fout,"%13s - %13s  ",amt1,amt2) ;
      for (j=0;j<N_MODES;j++)
      {
         fprintf (fout,"%4d  ",histogram[j][i]) ;
      }
      fprintf (fout,"\n") ;
   }

   /* The user requested a histogram file, create it */
   if ((fcsv=fopen(csvfname,"w")) == NULL)
   {
      perror("report file error:");
      exit(1) ;
   } ;

   /* print out the histogram values */
   fprintf (fcsv,"100-point histogram for a spreadsheet\n") ;
   fprintf (fcsv,"Making a histogram is left up to you! :)\n") ;
   fprintf (fcsv,"from,to,mode1,mode2,mode3\n") ;
   for (i=0;i<100;i++)
   {
      fprintf (fcsv,"%8.0f,%8.0f,",hmin+(hmax-hmin)*i/100,hmin+(hmax-hmin)*(i+1)/100) ;
      for (j=0;j<N_MODES;j++)
      {
         fprintf (fcsv,"%4d,",histogram[j][i]) ;
      }
      fprintf (fcsv,"\n") ;
   }
   fclose(fcsv) ;
   

} ;

/****************************************************************************
* Routines : report_summaries
* Gazintas : summaries
* IOs      : None
* Returns  : None
* Globals  : params
*
* This is the routine to report back the pass/fail summaries of each simulation
****************************************************************************/
void report_summaries(SUMMARY* summaries,RESULTS* results,FILE* fout)
{
   uint32 i ;
   char amt[16] ;

   /* if this is the first call, compute the mean value of each mode */
   /* I know first time if the mean value is zero.                   */
   if (summaries[0].mean_left==0)
   {
      /* To get the mean result, sort by each result and get middle point */
      /* sort by Mode 0 first */
      qsort(results,params.num_sims,sizeof(RESULTS),(void*)res0sort) ;
      summaries[0].mean_left=results[params.num_sims/2].estate[0] ;

      /* sort by Mode 1 next */
      qsort(results,params.num_sims,sizeof(RESULTS),(void*)res1sort) ;
      summaries[1].mean_left=results[params.num_sims/2].estate[1] ;

      /* sort by Mode 2 next */
      qsort(results,params.num_sims,sizeof(RESULTS),(void*)res2sort) ;
      summaries[2].mean_left=results[params.num_sims/2].estate[2] ;
   } ;

   for (i=0;i<N_MODES;i++)
   {

     /* end with the super-summary of the results */
     fprintf (fout,"\nMode%d: %d of %d successes for %d%% success rate.\n",i,summaries[i].successes,params.num_sims,summaries[i].successes*100/params.num_sims) ;
     itoc(amt,(int32)summaries[i].max_left,10,'$',TRUE) ;
     fprintf (fout,"Max  estate and end %13s\n",amt) ;
     itoc(amt,(int32)summaries[i].min_left,10,'$',TRUE) ;
     fprintf (fout,"Min  estate and end %13s\n",amt) ;
     itoc(amt,(int32)summaries[i].avg_left,10,'$',TRUE) ;
     fprintf (fout,"Avg  estate and end %13s\n",amt) ;
     itoc(amt,(int32)summaries[i].mean_left,10,'$',TRUE) ;
     fprintf (fout,"Mean estate and end %13s\n",amt) ;
   } ;
} ;


