#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <conio.h>
#include <ctype.h>

#include "def.h"
#include "Simulation.h"
#include "params.h"
#include "ini.h"
#include "util.h"

uint8 INI_OptInt (uint32* param, char* section, char* key         ) ;
uint8 INI_OptFlt (float*  param, char* section, char* key         ) ;
uint8 INI_OptStr (char*   param, char* section, char* key, int len) ;
uint8 INI_OptBool(uint8*  param, char* section, char* key         ) ;

PARAMS params     ; /* parameter structure  */
uint8  Perror[36] ; /* error flag array     */
BATCH  BatchDat   ; /* Batch data           */
uint8  Berror[11] ; /* Batch error flags    */
uint8  nbatch=0   ; /* # of batch runs      */
uint8 nbatchs=1   ; /* # of batch SSIs      */
uint8 nbatchc=1   ; /* # of batch CPIs      */
/****************************************************************************
* Routne   : init_params
* Gazintas : None
* IOs      : None
* Returns  : Nothing
* Globals  : params
*
* This routine sets all the entries in the parameter block to benign values.
****************************************************************************/
void init_params(void)
{
   /* start off with default values for everything */
   params.start_amount            = 100   ;
   params.start_year              = 2030  ;
   params.start_age               = 65    ;
   params.end_age                 = 95    ;
   params.mkt_return_avg          = 0     ;
   params.mkt_return_range        = 0     ;

   params.age_ror_phase2          = 0     ;
   params.mkt_return_avg2         = 0     ;
   params.mkt_return_range2       = 0     ;

   params.age_ror_phase3          = 0     ;
   params.mkt_return_avg3         = 0     ;
   params.mkt_return_range3       = 0     ;

   params.cat1_expense_name[0]    = 0     ;
   params.cat1_expense_amount_min = 0     ;
   params.cat1_expense_amount_max = 0     ;
   params.cat1_expense_inflation  = 0     ;

   params.cat2_expense_name[0]    = 0     ;
   params.cat2_expense_amount_min = 0     ;
   params.cat2_expense_amount_max = 0     ;
   params.cat2_expense_inflation  = 0     ;

   params.cat3_expense_name[0]    = 0     ;
   params.cat3_expense_amount_min = 0     ;
   params.cat3_expense_amount_max = 0     ;
   params.cat3_expense_inflation  = 0     ;

   params.soc_sec_age_start       = 62    ;
   params.soc_sec_amt             = 0     ;
   params.soc_sec_inflation       = 0     ;
   params.soc_sec_use             = FALSE ;

   params.num_sims                = 500   ;
   params.logs                    = FALSE ;
   params.detail_summary          = FALSE ;
   params.histogram               = FALSE ;
   params.debug                   = FALSE ;

   strcpy(params.SummaryFile,  "Summary.txt"  ) ;
   strcpy(params.HistogramFile,"Histogram.csv") ;

   params.first= FALSE ;

   /* Now for the batch mode data */
   BatchDat.BStash[0]=0.0 ;
   BatchDat.BStash[1]=0.0 ;
   BatchDat.BStash[2]=0.0 ;
   BatchDat.BStash[3]=0.0 ;
   BatchDat.BStash[4]=0.0 ;

   BatchDat.AltCPI     = 0.0   ;
   BatchDat.UseAltCPI1 = FALSE ;
   BatchDat.UseAltCPI2 = FALSE ;
   BatchDat.UseAltCPI3 = FALSE ;
   BatchDat.BothSSIs   = FALSE ;

} ;

/****************************************************************************
* Routne   : read_params
* Gazintas : batchmode: silences screen writes
* IOs      : file "fname"
* Returns  : fills in the structure "params" from file fname
* Globals  : params
*
* This routine reads the command file "fname" with all of the input data to
* run the simulation.  See the comments for the structure (top of file) for
* the meanings of each variable.
****************************************************************************/
uint8 read_params (char* fname,int batchmode)
{
   int error ; /* error flag      */
   int i     ; /* general counter */

   if (!batchmode) printf("\nReading parameter file \"%s\"...\n",fname) ;

   /* go open and parse the parameter file */
   error = INI_ParseFile(fname);
   if (error!=INI_ERR_NONE) return (error) ;

   /* all is well, now go get parameters, just go through the list */
   Perror[ 0] = FALSE ;
   Perror[ 1] = INI_OptFlt (&params.start_amount           ,"WinMonte","start_amount"           ) ;
   Perror[ 2] = INI_OptInt (&params.start_year             ,"WinMonte","start_year"             ) ;
   Perror[ 3] = INI_OptInt (&params.start_age              ,"WinMonte","start_age"              ) ;
   Perror[ 4] = INI_OptInt (&params.end_age                ,"WinMonte","end_age"                ) ;
   Perror[ 5] = INI_OptFlt (&params.mkt_return_avg         ,"WinMonte","mkt_return_avg1"        ) ;
   Perror[ 6] = INI_OptFlt (&params.mkt_return_range       ,"WinMonte","mkt_return_range1"      ) ;

   Perror[ 7] = INI_OptInt (&params.age_ror_phase2         ,"WinMonte","age_ror_phase2"         ) ;
   Perror[ 8] = INI_OptFlt (&params.mkt_return_avg2        ,"WinMonte","mkt_return_avg2"        ) ;
   Perror[ 9] = INI_OptFlt (&params.mkt_return_range2      ,"WinMonte","mkt_return_range2"      ) ;

   Perror[10] = INI_OptInt (&params.age_ror_phase3         ,"WinMonte","age_ror_phase3"         ) ;
   Perror[11] = INI_OptFlt (&params.mkt_return_avg3        ,"WinMonte","mkt_return_avg3"        ) ;
   Perror[12] = INI_OptFlt (&params.mkt_return_range3      ,"WinMonte","mkt_return_range3"      ) ;

   Perror[13] = INI_OptStr ( params.cat1_expense_name      ,"WinMonte","cat1_expense_name"   ,20) ;
   Perror[14] = INI_OptFlt (&params.cat1_expense_amount_min,"WinMonte","cat1_expense_amount_min") ;
   Perror[15] = INI_OptFlt (&params.cat1_expense_amount_max,"WinMonte","cat1_expense_amount_max") ;
   Perror[16] = INI_OptFlt (&params.cat1_expense_inflation ,"WinMonte","cat1_expense_inflation" ) ;

   Perror[17] = INI_OptStr ( params.cat2_expense_name      ,"WinMonte","cat2_expense_name"   ,20) ;
   Perror[18] = INI_OptFlt (&params.cat2_expense_amount_min,"WinMonte","cat2_expense_amount_min") ;
   Perror[19] = INI_OptFlt (&params.cat2_expense_amount_max,"WinMonte","cat2_expense_amount_max") ;
   Perror[20] = INI_OptFlt (&params.cat2_expense_inflation ,"WinMonte","cat2_expense_inflation" ) ;

   Perror[21] = INI_OptStr ( params.cat3_expense_name      ,"WinMonte","cat3_expense_name"   ,20) ;
   Perror[22] = INI_OptFlt (&params.cat3_expense_amount_min,"WinMonte","cat3_expense_amount_min") ;
   Perror[23] = INI_OptFlt (&params.cat3_expense_amount_max,"WinMonte","cat3_expense_amount_max") ;
   Perror[24] = INI_OptFlt (&params.cat3_expense_inflation ,"WinMonte","cat3_expense_inflation" ) ;

   Perror[25] = INI_OptInt (&params.soc_sec_age_start      ,"WinMonte","soc_sec_age_start"      ) ;
   Perror[26] = INI_OptFlt (&params.soc_sec_amt            ,"WinMonte","soc_sec_amt"            ) ;
   Perror[27] = INI_OptFlt (&params.soc_sec_inflation      ,"WinMonte","soc_sec_inflation"      ) ;
   Perror[28] = INI_OptBool(&params.soc_sec_use            ,"WinMonte","soc_sec_use"            ) ;

   Perror[29] = INI_OptInt (&params.num_sims               ,"WinMonte","num_sims"               ) ;
   Perror[30] = INI_OptBool(&params.logs                   ,"WinMonte","logs"                   ) ;
   Perror[31] = INI_OptBool(&params.detail_summary         ,"WinMonte","detail_summary"         ) ;
   Perror[32] = INI_OptBool(&params.histogram              ,"WinMonte","histogram"              ) ;
   Perror[33] = INI_OptBool(&params.debug                  ,"WinMonte","debug"                  ) ;

   Perror[34] = INI_OptStr ( params.SummaryFile            ,"WinMonte","SummaryFile"   ,MAX_PATH) ;
   Perror[35] = INI_OptStr ( params.HistogramFile          ,"WinMonte","HistogramFile" ,MAX_PATH) ;

   /* Now for the batch mode data */
   Berror[ 0] = FALSE ;
   Berror[ 1] = INI_OptFlt (&BatchDat.BStash[0] ,"WinMonte","Batch_Stash0") ;
   Berror[ 2] = INI_OptFlt (&BatchDat.BStash[1] ,"WinMonte","Batch_Stash1") ;
   Berror[ 3] = INI_OptFlt (&BatchDat.BStash[2] ,"WinMonte","Batch_Stash2") ;
   Berror[ 4] = INI_OptFlt (&BatchDat.BStash[3] ,"WinMonte","Batch_Stash3") ;
   Berror[ 5] = INI_OptFlt (&BatchDat.BStash[4] ,"WinMonte","Batch_Stash4") ;
   Berror[ 6] = INI_OptFlt (&BatchDat.AltCPI    ,"WinMonte","AltCPI"      ) ;

   Berror[ 7] = INI_OptBool(&BatchDat.UseAltCPI1,"WinMonte","UseAltCPI1"  ) ;
   Berror[ 8] = INI_OptBool(&BatchDat.UseAltCPI2,"WinMonte","UseAltCPI2"  ) ;
   Berror[ 9] = INI_OptBool(&BatchDat.UseAltCPI3,"WinMonte","UseAltCPI3"  ) ;
   Berror[10] = INI_OptBool(&BatchDat.BothSSIs  ,"WinMonte","BothSSIs"    ) ;

   /* After all that, see if any errors */
   error=FALSE;
   for (i=1;i<36;i++)
   {
      Perror[0] |=Perror[i] ;
   }
   for (i=1;i<9;i++)
   {
      Perror[0] |=Berror[i] ;
      Berror[0] |=Berror[i] ;
   }
   if (Perror[0]) return (PARAMS_ERR_BOX) ;

   /* some quick sanity checks */
   if (params.cat1_expense_amount_min>params.cat1_expense_amount_max)
   {
      Perror[14]=TRUE ;
      Perror[15]=TRUE ;
      return (PARAMS_ERR_MINMAX) ;
   } ;

   if (params.cat2_expense_amount_min>params.cat2_expense_amount_max)
   {
      Perror[18]=TRUE ;
      Perror[19]=TRUE ;
      return (PARAMS_ERR_MINMAX) ;
   } ;

   if (params.cat3_expense_amount_min>params.cat3_expense_amount_max)
   {
      Perror[22]=TRUE ;
      Perror[23]=TRUE ;
      return (PARAMS_ERR_MINMAX) ;
   } ;

   /* Use Perror[0] as a summary of all errors */
   for(i=1;i<36;i++) Perror[0]|=Perror[i] ;
   return(0) ;
} ;

/****************************************************************************
* Routne   : write_params
* Gazintas : fout : File to write to.
* IOs      : None
* Returns  : nothing
* Globals  : params
*
* This routine writes out the simulation parameters to a file.
****************************************************************************/
void write_params(FILE* fout)
{
   char amt[16] ;

   /* to get this far, things must have worked out OK, so print the results */
   /* and return with a success code.                                       */
   fprintf (fout,"The Integral Monte Carlo simulator!\n") ;
   fprintf (fout,"The simulator that integrates spending with rate of return.\n\n") ;

   fprintf (fout,"Source information:\n") ;
   fprintf (fout,"================================\n") ;
   itoc(amt,(int32)params.start_amount,10,'$',TRUE) ;
   fprintf (fout,"start_amount............ %s\n"     ,amt                            ) ;
   fprintf (fout,"start_year.............. %d\n"     ,params.start_year              ) ;
   fprintf (fout,"start_age............... %d\n"     ,params.start_age               ) ;
   fprintf (fout,"end_age................. %d\n"     ,params.end_age                 ) ;
   fprintf (fout,"================================\n") ;
   fprintf (fout,"mkt_return_avg1......... %6.3f%%\n",params.mkt_return_avg          ) ;
   fprintf (fout,"mkt_return_range1....... %6.3f%%\n",params.mkt_return_range        ) ;
   fprintf (fout,"age_ror_phase2.......... %d\n"     ,params.age_ror_phase2          ) ;
   fprintf (fout,"mkt_return_avg2......... %6.3f%%\n",params.mkt_return_avg2         ) ;
   fprintf (fout,"mkt_return_range2....... %6.3f%%\n",params.mkt_return_range2       ) ;
   fprintf (fout,"age_ror_phase3.......... %d\n"     ,params.age_ror_phase3          ) ;
   fprintf (fout,"mkt_return_avg3......... %6.3f%%\n",params.mkt_return_avg3         ) ;
   fprintf (fout,"mkt_return_range3....... %6.3f%%\n",params.mkt_return_range3       ) ;
   fprintf (fout,"================================\n") ;
   fprintf (fout,"cat1_expense_name....... %s\n"     ,params.cat1_expense_name       ) ;
   itoc(amt,(int32)params.cat1_expense_amount_min,10,'$',TRUE) ;
   fprintf (fout,"cat1_expense_amount_min. %s\n"     ,amt                            ) ;
   itoc(amt,(int32)params.cat1_expense_amount_max,10,'$',TRUE) ;
   fprintf (fout,"cat1_expense_amount_max. %s\n"     ,amt                            ) ;
   fprintf (fout,"cat1_expense_inflation.. %2.3f%%\n",params.cat1_expense_inflation  ) ;
   fprintf (fout,"================================\n") ;
   fprintf (fout,"cat2_expense_name....... %s\n"     ,params.cat2_expense_name       ) ;
   itoc(amt,(int32)params.cat2_expense_amount_min,10,'$',TRUE) ;
   fprintf (fout,"cat2_expense_amount_min. %s\n"     ,amt                            ) ;
   itoc(amt,(int32)params.cat2_expense_amount_max,10,'$',TRUE) ;
   fprintf (fout,"cat2_expense_amount_max. %s\n"     ,amt                            ) ;
   fprintf (fout,"cat2_expense_inflation.. %2.3f%%\n",params.cat2_expense_inflation  ) ;
   fprintf (fout,"================================\n") ;
   fprintf (fout,"cat3_expense_name....... %s\n"     ,params.cat3_expense_name       ) ;
   itoc(amt,(int32)params.cat3_expense_amount_min,10,'$',TRUE) ;
   fprintf (fout,"cat3_expense_amount_min. %s\n"     ,amt                            ) ;
   itoc(amt,(int32)params.cat3_expense_amount_max,10,'$',TRUE) ;
   fprintf (fout,"cat3_expense_amount_max. %s\n"    ,amt                             ) ;
   fprintf (fout,"cat3_expense_inflation.. %2.3f%%\n",params.cat3_expense_inflation  ) ;
   fprintf (fout,"================================\n") ;
   fprintf (fout,"soc_sec_age_start....... %2d\n"    ,params.soc_sec_age_start       ) ;
   itoc(amt,(int32)params.soc_sec_amt,10,'$',TRUE) ;
   fprintf (fout,"soc_sec_amt............. %s\n"     ,amt                            ) ;
   fprintf (fout,"soc_sec_inflation....... %2.3f%%\n",params.soc_sec_inflation       ) ;
   fprintf (fout,"soc_sec_use............. %s\n"     ,params.soc_sec_use?"TRUE":"FALSE") ;
   fprintf (fout,"================================\n") ;
   fprintf (fout,"num_sims................ %d\n"     ,params.num_sims                ) ;
   fprintf (fout,"logs.................... %s\n"     ,params.logs?"TRUE":"FALSE"     ) ;
   fprintf (fout,"detail_summary.......... %s\n"     ,params.detail_summary?"TRUE":"FALSE") ;
   fprintf (fout,"histogram............... %s\n"     ,params.histogram?"TRUE":"FALSE") ;
   fprintf (fout,"debug................... %s\n"     ,params.debug?"TRUE":"FALSE"    ) ;
   fprintf (fout,"================================\n") ;
   fprintf (fout,"Summary file............ %s\n"     ,params.SummaryFile             ) ;
   fprintf (fout,"Histogram file.......... %s\n"     ,params.HistogramFile           ) ;
   fprintf (fout,"================================\n\n") ;
   itoc(amt,(int32)BatchDat.BStash[0],10,'$',TRUE) ;
   fprintf (fout,"Batch year 0 Stash...... %s\n"     ,amt                            ) ;
   itoc(amt,(int32)BatchDat.BStash[1],10,'$',TRUE) ;
   fprintf (fout,"Batch year 1 Stash...... %s\n"     ,amt                            ) ;
   itoc(amt,(int32)BatchDat.BStash[2],10,'$',TRUE) ;
   fprintf (fout,"Batch year 2 Stash...... %s\n"     ,amt                            ) ;
   itoc(amt,(int32)BatchDat.BStash[3],10,'$',TRUE) ;
   fprintf (fout,"Batch year 3 Stash...... %s\n"     ,amt                            ) ;
   itoc(amt,(int32)BatchDat.BStash[4],10,'$',TRUE) ;
   fprintf (fout,"Batch year 4 Stash...... %s\n"     ,amt                            ) ;
   fprintf (fout,"Batch Alternate CPI..... %2.3f%%\n",BatchDat.AltCPI                ) ;
   fprintf (fout,"Batch Alt CPI Cat 1..... %s\n"     ,BatchDat.UseAltCPI1?"TRUE":"FALSE") ;
   fprintf (fout,"Batch Alt CPI Cat 2..... %s\n"     ,BatchDat.UseAltCPI2?"TRUE":"FALSE") ;
   fprintf (fout,"Batch Alt CPI Cat 3..... %s\n"     ,BatchDat.UseAltCPI3?"TRUE":"FALSE") ;
   fprintf (fout,"Batch SSI On/Off........ %s\n"     ,BatchDat.BothSSIs?"TRUE":"FALSE"  ) ;
   fprintf (fout,"================================\n\n") ;

} ;

/****************************************************************************
* Routne   : save_params
* Gazintas : fname : filename to write to
* IOs      : None
* Returns  : nothing
* Globals  : params
*
* This routine saves the parameters in .ini format for reading back later
****************************************************************************/
void save_params(char* fname)
{
   FILE* fout ; /* the file to write to */

   if ((fout=fopen(fname,"w")) == NULL)
   {
       perror("Parameter file error:");
   }

   /* write out all the parameters */
   fprintf (fout,"# WinMonte Simulation parameter file.  Saved by WinMonte\n") ;
   fprintf (fout,"# OK to manually edit this file, see docs for parameter descriptions.\n") ;
   fprintf (fout,"#\n") ;
   fprintf (fout,"[WinMonte]\n") ;

   fprintf (fout,"#============================================\n"                  ) ;
   fprintf (fout,"start_amount            = %1.0f\n" ,params.start_amount           ) ;
   fprintf (fout,"start_year              = %d\n"    ,params.start_year             ) ;
   fprintf (fout,"start_age               = %d\n"    ,params.start_age              ) ;
   fprintf (fout,"end_age                 = %d\n"    ,params.end_age                ) ;
   fprintf (fout,"#============================================\n"                  ) ;
   fprintf (fout,"mkt_return_avg1         = %6.3f\n",params.mkt_return_avg          ) ;
   fprintf (fout,"mkt_return_range1       = %6.3f\n",params.mkt_return_range        ) ;
   fprintf (fout,"age_ror_phase2          = %d\n"   ,params.age_ror_phase2          ) ;
   fprintf (fout,"mkt_return_avg2         = %6.3f\n",params.mkt_return_avg2         ) ;
   fprintf (fout,"mkt_return_range2       = %6.3f\n",params.mkt_return_range2       ) ;
   fprintf (fout,"age_ror_phase3          = %d\n"   ,params.age_ror_phase3          ) ;
   fprintf (fout,"mkt_return_avg3         = %6.3f\n",params.mkt_return_avg3         ) ;
   fprintf (fout,"mkt_return_range3       = %6.3f\n",params.mkt_return_range3       ) ;
   fprintf (fout,"#============================================\n"                  ) ;
   fprintf (fout,"cat1_expense_name       = %s\n"   ,params.cat1_expense_name       ) ;
   fprintf (fout,"cat1_expense_amount_min = %1.0f\n",params.cat1_expense_amount_min ) ;
   fprintf (fout,"cat1_expense_amount_max = %1.0f\n",params.cat1_expense_amount_max ) ;
   fprintf (fout,"cat1_expense_inflation  = %2.3f\n",params.cat1_expense_inflation  ) ;
   fprintf (fout,"#============================================\n"                  ) ;
   fprintf (fout,"cat2_expense_name       = %s\n"   ,params.cat2_expense_name       ) ;
   fprintf (fout,"cat2_expense_amount_min = %1.0f\n",params.cat2_expense_amount_min ) ;
   fprintf (fout,"cat2_expense_amount_max = %1.0f\n",params.cat2_expense_amount_max ) ;
   fprintf (fout,"cat2_expense_inflation  = %2.3f\n",params.cat2_expense_inflation  ) ;
   fprintf (fout,"#============================================\n"                  ) ;
   fprintf (fout,"cat3_expense_name       = %s\n"   ,params.cat3_expense_name       ) ;
   fprintf (fout,"cat3_expense_amount_min = %1.0f\n",params.cat3_expense_amount_min ) ;
   fprintf (fout,"cat3_expense_amount_max = %1.0f\n",params.cat3_expense_amount_max ) ;
   fprintf (fout,"cat3_expense_inflation  = %2.3f\n",params.cat3_expense_inflation  ) ;
   fprintf (fout,"#============================================\n"                  ) ;
   fprintf (fout,"soc_sec_age_start       = %2d\n"  ,params.soc_sec_age_start       ) ;
   fprintf (fout,"soc_sec_amt             = %1.0f\n",params.soc_sec_amt             ) ;
   fprintf (fout,"soc_sec_inflation       = %2.3f\n",params.soc_sec_inflation       ) ;
   fprintf (fout,"soc_sec_use             = %s\n"   ,params.soc_sec_use?"Y":"N") ;
   fprintf (fout,"#============================================\n"                  ) ;
   fprintf (fout,"num_sims                = %d\n"   ,params.num_sims                ) ;
   fprintf (fout,"logs                    = %s\n"   ,params.logs?"Y":"N"            ) ;
   fprintf (fout,"detail_summary          = %s\n"   ,params.detail_summary?"Y":"N"  ) ;
   fprintf (fout,"histogram               = %s\n"   ,params.histogram?"Y":"N"       ) ;
   fprintf (fout,"debug                   = %s\n"   ,params.debug?"Y":"N"           ) ;
   fprintf (fout,"#============================================\n"                  ) ;
   fprintf (fout,"SummaryFile             = %s\n"   ,params.SummaryFile             ) ;
   fprintf (fout,"HistogramFile           = %s\n"   ,params.HistogramFile           ) ;
   fprintf (fout,"#============================================\n"                  ) ;
   fprintf (fout,"Batch_Stash0            = %1.0f\n",BatchDat.BStash[0]             ) ;
   fprintf (fout,"Batch_Stash1            = %1.0f\n",BatchDat.BStash[1]             ) ;
   fprintf (fout,"Batch_Stash2            = %1.0f\n",BatchDat.BStash[2]             ) ;
   fprintf (fout,"Batch_Stash3            = %1.0f\n",BatchDat.BStash[3]             ) ;
   fprintf (fout,"Batch_Stash4            = %1.0f\n",BatchDat.BStash[4]             ) ;
   fprintf (fout,"AltCPI                  = %2.3f\n",BatchDat.AltCPI                ) ;
   fprintf (fout,"UseAltCPI1              = %s\n"   ,BatchDat.UseAltCPI1?"Y":"N"    ) ;
   fprintf (fout,"UseAltCPI2              = %s\n"   ,BatchDat.UseAltCPI2?"Y":"N"    ) ;
   fprintf (fout,"UseAltCPI3              = %s\n"   ,BatchDat.UseAltCPI3?"Y":"N"    ) ;
   fprintf (fout,"BothSSIs                = %s\n"   ,BatchDat.BothSSIs?"Y":"N"      ) ;

   fclose(fout) ;

} ;

/****************************************************************************
* Routne   : INI_OptInt
* Gazintas : section = Section and key to look for in .ini file
*            key     = Key to look for in .ini file
* IOs      : param   = value to replace if found in .ini file
* Returns  : True if error
* Globals  : None
*
* This routine looks for the specified section/key for an assumed integer
* value and if found replaces the param with this new value.  If the section/
* key is not found, the parameter is left alone.  I just use atoi, so no real
* error checking is done.
****************************************************************************/
uint8 INI_OptInt(uint32* param, char* section, char* key)
{
   char*  str   ;

   /* look for section/key, if not found, return without changing the parameter */
   str =INI_GetString(section,key) ;
   if (str==NULL) return (0) ;

   /* we got something, convert it to an integer */
   /* no error possible                          */
   return atoi_e(param,str) ;

} ;

/****************************************************************************
* Routne   : INI_OptFlt
* Gazintas : section = Section and key to look for in .ini file
*            key     = Key to look for in .ini file
* IOs      : param   = value to replace if found in .ini file
* Returns  : True if error
* Globals  : None
*
* This routine looks for the specified section/key for an assumed float number
* value and if found replaces the param with this new value.  If the section/
* key is not found, the parameter is left alone.  I just use atoi, so no real
* error checking is done.
****************************************************************************/
uint8 INI_OptFlt(float* param, char* section, char* key)
{
   char*  str   ;

   /* look for section/key, if not found, return without changing the parameter */
   str =INI_GetString(section,key) ;
   if (str==NULL) return (0) ;

   /* we got something, convert it to a float */
   /* no error possible                       */
   return atof_e(param,str) ;

   return 0;

} ;

/****************************************************************************
* Routne   : INI_OptStr
* Gazintas : section = Section and key to look for in .ini file
*            key     = Key to look for in .ini file
*            len     = Dest string length
* IOs      : param   = value to replace if found in .ini file
* Returns  : True if error
* Globals  : None
*
* This routine looks for the specified section/key for an assumed string
* and if found replaces the param with this new value.  If the section/
* key is not found, the parameter is left alone.  I just use atoi, so no real
* error checking is done.
****************************************************************************/
uint8 INI_OptStr(char* param, char* section, char* key, int len)
{
   char*  str   ;

   /* look for section/key, if not found, return without changing the parameter */
   str =INI_GetString(section,key) ;
   if (str==NULL) return (0) ;

   /* we got something, coopy it over */
   /* no error possible               */
   strncpy(param,str,len) ;
   return 0;

} ;

/****************************************************************************
* Routne   : INI_OptBool
* Gazintas : section = Section and key to look for in .ini file
*            key     = Key to look for in .ini file
* IOs      : param   = value to replace if found in .ini file
* Returns  : True if error
* Globals  : None
*
* This routine looks for the specified section/key for an assumed boolean
* value and if found replaced the param with this new value.  If the section/
* key is not found, the parameter is left alone.  If in the file TRUE is
* if the test file contains TRUE or Yes (just looking at the first letter)
* FALSE is returned for FALSE or No (again, only looknig at first letter.
****************************************************************************/
uint8 INI_OptBool(uint8* param, char* section, char* key)
{
   char* value ;

   /* look for section/key, if not found, return without changing the parameter */
   value =INI_GetString(section,key) ;
   if (value==NULL) return (0) ;

   /* We have the parameter, replace it */
   value[0]=toupper(value[0]) ;

   if ((value[0]=='Y')||(value[0]=='T'))
   {
      *param = TRUE  ;
      return (0) ;
   }

   if ((value[0]=='N')||(value[0]=='F'))
   {
      *param = FALSE  ;
      return (0) ;
   }

   return 1;

} ;


