#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "def.h"
#include "Simulation.h"
#include "params.h"
#include "modes.h"

/****************************************************************************
* Routne   : Mode0
* Gazintas : yearly - consistent things for each year regardless of mode
* IOs      : sim[0] - Mode0 accumulator data
* Returns  : None
* Globals  : params - Simulation parameters
*
* This routine implements Mode 0.  This is the "classic" way of running a
* Monte Carlo simulation, where the spending for each year is fixed, only
* increasing by the rate of inflation.  In this mode, the annual spend for each
* catetory is the max value, the min value is not used.
*
****************************************************************************/
void Mode0(SIMD *sim,YEARLY *yearly)
{
   /* start with previous years final */
   sim->start=sim->final ;

   /* compute the dollar amount of the rate of return for each spending mode */
   sim->rord=sim->start*yearly->rorp/100 ;

   /* I let the balance go negative to see how far the "failures" go, but in */
   /* doing so, RoR better be zero when there isn't a balance.               */
   if (sim->start<0.5) sim->rord=0.0 ;

   /* for this mode, pretty simple, the actual spends are just the max       */
   /* spends for each category.                                              */
   sim->spend1act = yearly->cat1max ;
   sim->spend2act = yearly->cat2max ;
   sim->spend3act = yearly->cat3max ;

   /* All known now, just do the math */
   sim->final = sim->start
              + sim->rord
              - sim->spend1act
              - sim->spend2act
              - sim->spend3act
              + yearly->soc_sec_adder ;

   /* if logs are turned on, show all this data */
   if (params.logs)
   {
      //fprintf (fout,"%8.0f %8.0f %8.0f %8.0f %8.0f %8.0f ",
      //sim->start,rord,yearly->cat1max,yearly->cat2max,yearly->cat3max,sim->final) ;
   } ;

}

/****************************************************************************
* Routne   : Mode1
* Gazintas : yearly - Consistent things for each year regardless of mode
* IOs      : sim[1] - Mode1 accumulator data
* Returns  : None
* Globals  : params - Simulation parameters
*
* This routine implements Mode 1.  This mode differs from the classic method
* in that there is a min and max amount for each category.  The minimum amount
* is necessary spending not matter what the return.  The max amount is the
* desired spending if the RoR supports it.  The idea is that you spend the full
* amount on good years, but hunker down and spend the minimum amount on years
* where the stock market is not so kind.
*
****************************************************************************/
void Mode1(SIMD *sim,YEARLY *yearly)
{
   float spend_max,spend_min           ; /* min and max total spend           */

   /* start with previous years final */
   sim->start=sim->final ;

   /* compute the dollar amount of the rate of return for each spending mode  */
   sim->rord=sim->start*yearly->rorp/100 ;

   /* I let the balance go negative to see how far the "failures" go, but in  */
   /* doing so, RoR better be zero when there isn't a balance.                */
   if (sim->start<0.5) sim->rord=0.0 ;

   /* start off assuming max available spending for Mode 2                    */
   sim->spend1act=yearly->cat1max ;      /* start off by copying ideal (max) values */
   sim->spend2act=yearly->cat2max ;      /* start off by copying ideal (max) values */
   sim->spend3act=yearly->cat3max ;      /* start off by copying ideal (max) values */

   /* Compute the total max and min spends for comparisons later.             */
   spend_max=yearly->cat1max
            +yearly->cat2max
            +yearly->cat3max ;
   spend_min=yearly->cat1min
            +yearly->cat2min
            +yearly->cat3min ;

   if (spend_max>sim->rord)         /* if spend is greater than RoR, need to limit it */
   {                                /* max spends need no logic as they're default    */

      if (sim->rord<spend_min) /* if RoR is less than min spend, set all to minimums  */
      {
         sim->spend1act = yearly->cat1min ;
         sim->spend2act = yearly->cat2min ;
         sim->spend3act = yearly->cat3min ;
      }
      else /* less than max spend, but more than minimum, so allocate proportionally */
      {
         /* local variables */
         float avail, span1,span2,span3,ratio ;

         /* this is the amount available over minimums */
         avail= sim->rord-spend_min ;

         /* to figure out spend per category, get the span for each one */
         span1= yearly->cat1max-yearly->cat1min ;
         span2= yearly->cat2max-yearly->cat2min ;
         span3= yearly->cat3max-yearly->cat3min ;

         /* for debugging, show the intermediate values */
         if (params.debug) fprintf(flog,"min,max,rord2 %8.0f %8.0f %8.0f\n",spend_min,spend_max,sim->rord) ;
         if (params.debug) fprintf(flog,"avail %8.0f span1-3  %8.0f %8.0f %8.0f\n",avail, span1, span2, span3) ;

         /* This ratio is the amount available compared to the total spend needed */
         ratio=avail/(span1+span2+span3) ;

         /* The spend is the minimum for each category plus the ratio'd span for each */
         sim->spend1act = yearly->cat1min+ratio*span1 ;
         sim->spend2act = yearly->cat2min+ratio*span2 ;
         sim->spend3act = yearly->cat3min+ratio*span3 ;

         /* more debugging of intermediate values */
         if (params.debug) fprintf(flog,"ratio %8.5f spend1-3 %8.0f %8.0f %8.0f\n",ratio, ratio*span1, ratio*span2, ratio*span3) ;
      }
   }

   /* All known now, just do the math */
   sim->final = sim->start
              + sim->rord
              - sim->spend1act
              - sim->spend2act
              - sim->spend3act
              + yearly->soc_sec_adder ;

   /* if logs are turned on, show all this data */
   if (params.logs)
   {
      //fprintf (fout,"%8.0f %8.0f %8.0f %8.0f %8.0f %8.0f ",
      //sim->start,rord,sim->spend1act,sim->spend2act,sim->spend3act,sim->final) ;
   } ;
}

/****************************************************************************
* Routne   : Mode2
* Gazintas : rorp
* IOs      : sim[2]
* Returns  : None
* Globals  : params
*
* This routine implements Mode 2.  This mode tries to emulate the "keep N years
* of necessary spending in cash" so that you don't have to sell stocks during
* down years.  N is the #define M2Years.  In this mode, we pull M2Years worth
* of spending money up front into the cache variable.  At the beginning of each
* year we replenish the cache with up to M2Years worth of expenses or that years
* RoR, whichever is less.  If we don't have enough money, we then sell the
* stock for a loss.
*
****************************************************************************/
void Mode2(SIMD *sim,YEARLY *yearly)
{
   float spendtot ; /* total spending for the year */
   float refill   ; /* cache refill amount         */

   /* start with previous years final */
   sim->start=sim->final ;

   /* compute the dollar amount of the rate of return for each spending mode  */
   sim->rord=sim->start*yearly->rorp/100 ;

   /* I let the balance go negative to see how far the "failures" go, but in  */
   /* doing so, RoR better be zero when there isn't a balance.                */
   if (sim->start<0.5) sim->rord=0.0 ;

   /* for this mode, spending is pretty simple, the same as Mocde 0.  The     */
   /* actual spends are just the max spends for each category.                */
   /* Compute the total spend too, needed for later.                          */
   sim->spend1act = yearly->cat1max ;
   sim->spend2act = yearly->cat2max ;
   sim->spend3act = yearly->cat3max ;
   spendtot = sim->spend1act+sim->spend2act+sim->spend3act ;

   /* take the spending money from the cache              */
   sim->cache -= spendtot ;
   /* add Social Security to the cache if included        */
   sim->cache += yearly->soc_sec_adder ;

   /* Now to replenish the cache.  The logic is as follows:                   */
   /* The desired amount is M2Years times the current spendtot.               */
   /* If the RoR is greater than the replenishment anmount, refill to target. */
   /*  - Remainder goes back into investment pool.                            */
   /* If the RoR is less than the replensiment amount, but >0 refill AMAP.    */
   /*  - No remainder, so none extra into the investment pool.                */
   /* If the cache is >0, no further action required.                         */
   /* if the cache is <0, fill to 0 from investment pool.                     */
   /*  - This means we needed to pull money out of investments for living     */

   /* what we need to get the cache up to 3 years spending */
   refill = M2Years*spendtot - sim->cache ;

   /* if we have more investment income than needed to replenish the cache    */
   if (sim->rord>refill)
   {
      /* final investment is start plus income minus refill amont             */
      sim->final  = sim->start + sim->rord - refill ;
      sim->cache += refill ;
   } else if (sim->rord>0) /* income is less than refill amt, but >0          */
   {
      /* all income goes to cache, investment stays static                    */
      sim->cache += sim->rord ;
      sim->final  = sim->start ;
   }
   else /* we had an investment loss */
   {
      /* nothing to the cache, subtract the loss from the investment          */
      sim->final = sim->start + sim->rord ;
   } ;

   /* if the cache went negative, then pull money from the investments to get */
   /* back to zero.  This is effectivly pulling out investment money to live  */
   /* on.                                                                     */
   if (sim->cache<0)
   {
      sim->final = sim->final + sim->cache ;
      sim->cache = 0 ;
   }

   /* if logs are turned on, show all this data */
   if (params.logs)
   {
      //fprintf (fout,"%8.0f %8.0f %8.0f %8.0f %8.0f %8.0f ",
      //sim->start,rord,yearly->cat1max,yearly->cat2max,yearly->cat3max,sim->final) ;
   } ;
}


