#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include "def.h"
#include "ini.h"
#include "util.h"

/* ************************************************************************** */
/* INI file parser.  Originally written by Brian Ellis, fiddled with by Will  */
/* Beals thereafter.  Reads in a .ini file and then has functions to get      */
/* parameters from the file.  Public functions are:                           */
/* INI_ParseFile     : Reads in the entire .ini file parses, and stores it    */
/* INI_GetInteger    : Gets an integer value for a Section/Key                */
/* INI_GetString     : Gets a string value for a Section/Key                  */
/* INI_HasSection    : Checks to see if a section group exists                */
/* INI_Print         : For Debug, prings all parsed data                      */
/* INI_Iterate       : Sets up a scan for multiple keys of same name          */
/* INI_NextEntry     : Iterative call to get values for iterated keys         */
/* INI_StopIterating : Cleans up after iterating                              */
/* ************************************************************************** */

/* ************************************************************************** */
/* Data structures                                                            */
/* ************************************************************************** */
struct ini
{
   char *section    ; /* Section name */
   char *key        ; /* Key name     */
   char *value      ; /* key value    */
   struct ini *next ; /* Next triplet */
};

struct iterate
{
   char *section         ; /* Section to scan */
   char *key             ; /* Key to scan for */
   struct ini *startwith ; /* moving pointer  */
};

/* ************************************************************************** */
/* Base pointer for .ini structure linked list                                */
/* ************************************************************************** */
struct ini *entries = NULL;

/* ************************************************************************** */
/* Local routines                                                             */
/* ************************************************************************** */
static void ini_CreateKeyValuePair(char *section, char *key, char *value);

/* ************************************************************************** */
/* Public routines                                                            */
/* ************************************************************************** */

/* ************************************************************************** */
/* Routine  : INI_ParseFile                                                   */
/* Gazintas : filename - .ini file to open and parse                          */
/* IOs      : None                                                            */
/* Returns  : INI_ERR_NONE     : no error (all defined in ini.h)              */
/*            INI_ERR_FILEOPEN : .ini File Open error                         */
/*            INI_ERR_SYNTAX   : Syntax error in file                         */
/* Globals  : entries - the linked list of parameters                         */
/*                                                                            */
/* This routine reads in the .ini file, parses it, and stores the Section/    */
/* key/value triples in a malloc'd link list.  If any errors are encountered, */
/* this routine just returns with an error code, Brian's original printing    */
/* of error codes doesn't work in a Windows app! :)                           */
/* ************************************************************************** */
int INI_ParseFile(char *filename)
{
   FILE *fp           ; /* Input file          */
   int  linenum       ; /* Line # for tracking */
   char linebuf[1000] ; /* input line to parse */
   char section[100]  ; /* section name        */
   char key[100]      ; /* key name            */
   char value[200]    ; /* key value           */
   int  n             ; /* general counter     */

    /* open .ini file */
   fp = fopen(filename, "r");
   if (fp == NULL) return (INI_ERR_FILEOPEN) ;

   /* Initialize things */
   strcpy(section, "");
   linenum = 0;

   /* parse the whole file */
   while (!feof(fp))
   {
      /* read one line and trim leading and trailing white space */
      /* Exit loop if EOF                                        */
      if (fgets(linebuf, sizeof(linebuf), fp) == NULL)
      {
         break;
      }
      trimwhite(linebuf);
      ++linenum;

      /* skip blank lines, and comment lines */
      if ((linebuf[0] == '\0') || (linebuf[0] == '#'))
      {
         continue;
      }

      /* parse [section name], if success next line */
      n = sscanf(linebuf, "[%[^]]]", section);
      if (n == 1)
      {
         continue;
      }

      /* parse key = value, if success, store it and next line */
      n = sscanf(linebuf, "%[^=]=%[^\n]", key, value);
      if (n == 2)
      {
         trimwhite(key);
         trimwhite(value);
         ini_CreateKeyValuePair(section, key, value);
         continue;
      }

      /* if we get here, we have an invalid line */
      // printf("%s: line %d: syntax error\n", filename, linenum);
      return(INI_ERR_SYNTAX);
   }

   /* Only way to get here is EOF without errors */
   fclose(fp);
   return (INI_ERR_NONE) ;
}

/* ************************************************************************** */
/* Routine  : INI_GetInteger                                                  */
/* Gazintas : section : Section name to find                                  */
/*            key     : Key name to find                                      */
/* IOs      : None                                                            */
/* Returns  : value in integer form (or -1 if error)                          */
/* Globals  : entries - the linked list of parameters                         */
/*                                                                            */
/* This routine finds the Section/key and returns its integer value.          */
/* Any kind of error returns -1                                               */
/* ************************************************************************** */
int INI_GetInteger(char *section, char *key)
{
   char *cp;
   int retval;

   cp = INI_GetString(section, key);
   if (cp)
   {
      retval = atoi(cp);
   }
   else
   {
      retval = -1;
   }

   return retval;
}

/* ************************************************************************** */
/* Routine  : INI_GetString                                                   */
/* Gazintas : section : Section name to find                                  */
/*            key     : Key name to find                                      */
/* IOs      : None                                                            */
/* Returns  : pointer to string if it exists, NULL otherwise                  */
/* Globals  : entries - the linked list of parameters                         */
/*                                                                            */
/* This routine finds the Section/key and returns the value as a string.      */
/* If Section/Key is not found, NULL is returned.                             */
/* ************************************************************************** */
char *
INI_GetString(char *section, char *key)
{
   struct ini *pe;

   for (pe = entries; pe; pe = pe->next)
   {
      if (StrEqualNoCase(section, pe->section) &&
          StrEqualNoCase(key, pe->key))
      {
         return pe->value;
      }
   }
   return NULL;
}

/* ************************************************************************** */
/* Routine  : INI_HasSection                                                  */
/* Gazintas : section : Section name to find                                  */
/* IOs      : None                                                            */
/* Returns  : TRUE if section exists, FALSE otherwise                         */
/* Globals  : entries - the linked list of parameters                         */
/*                                                                            */
/* This routine checks to see if a section name exists                        */
/* ************************************************************************** */
int INI_HasSection(char *section)
{
   struct ini *pe;

   for (pe = entries; pe; pe = pe->next)
   {
      if (StrEqualNoCase(section, pe->section))
      {
         return TRUE;
      }
   }
   return FALSE;
}

/* ************************************************************************** */
/* Routine  : INI_Print                                                       */
/* Gazintas : None                                                            */
/* IOs      : None                                                            */
/* Returns  : None                                                            */
/* Globals  : entries - the linked list of parameters                         */
/*                                                                            */
/* This routine prints the structure to the console.  For debugging.          */
/* ************************************************************************** */
void INI_Print(void)
{
   struct ini *pe;

   for (pe = entries; pe; pe = pe->next)
   {
      printf("[%s] %s = %s\n", pe->section, pe->key, pe->value);
   }
}

/* ************************************************************************** */
/* Routine  : INI_Iterate                                                     */
/* Gazintas : section : Section name to find                                  */
/*            key     : Key name to find                                      */
/* IOs      : None                                                            */
/* Returns  : pointer temporary parsing structure                             */
/* Globals  : None                                                            */
/*                                                                            */
/* This routine initiates a search for multiple key entries for the same      */
/* section.  It returns a pointer to a structure to keep track of the search. */
/* This routine doesn't return the first find, just does the setup.           */
/* Not sure why the structure is voided, need to ask Brian.                   */
/* ************************************************************************** */
void * INI_Iterate(char *section, char *key)
{
   struct iterate *iterate;

   /* allocate storage */
   iterate = malloc(sizeof(*iterate));

   /* fill in fields */
   iterate->section   = section;
   iterate->key       = key;
   iterate->startwith = entries;

   return (void *)iterate;
}

/* ************************************************************************** */
/* Routine  : INI_NextEntry                                                   */
/* Gazintas : handle : Search structure (set up by INI_Iterate)               */
/* IOs      : None                                                            */
/* Returns  : pointer next entry (or NULL if done)                            */
/* Globals  : entries - the linked list of parameters                         */
/*                                                                            */
/* This routine returns the successive values for the Section/Key pair set up */
/* by INI_Iterate.  It keeps on returning them until no more exist and when   */
/* that happens, returns a NULL.                                              */
/* ************************************************************************** */
char* INI_NextEntry(void *handle)
{
   struct iterate *iterate = (struct iterate *)handle;
   struct ini *pe;

   /* step through the entries in the ini file */
   for (pe = iterate->startwith; pe; pe = pe->next)
   {
      /* right section? */
      if (StrEqualNoCase(pe->section, iterate->section))
      {
         /* rights key? */
         if (StrEqualNoCase(pe->key, iterate->key))
         {
            /* match. save our place, and return the value */
            iterate->startwith = pe->next;
            return pe->value;
         }
      }
   }

   /* only way to get here is no more matches. */
   return NULL;
}

/* ************************************************************************** */
/* Routine  : INI_StopIterating                                               */
/* Gazintas : handle : Search structure (set up by INI_Iterate)               */
/* IOs      : None                                                            */
/* Returns  : None                                                            */
/* Globals  : None                                                            */
/*                                                                            */
/* This routine cleans up after an iterative search, basically just frees the */
/* search structure.                                                          */
/* ************************************************************************** */
void INI_StopIterating(void *handle)
{
   struct iterate *iterate = (struct iterate *)handle;

   free(iterate);
}

/* ************************************************************************** */
/* *************************** Private Functions **************************** */
/* ************************************************************************** */

/* ************************************************************************** */
/* Routine  : ini_CreateKeyValuePair                                          */
/* Gazintas : section : Section name to create                                */
/*            key     : Key name to create                                    */
/*            value   : Value to store                                        */
/* IOs      : None                                                            */
/* Returns  : None                                                            */
/* Globals  : entries - the linked list of parameters                         */
/*                                                                            */
/* This routine cleans up after an iterative search, basically just frees the */
/* search structure.                                                          */
/* ************************************************************************** */
static void ini_CreateKeyValuePair(char *section, char *key, char *value)
{
   struct ini *entry;
   struct ini *pe;
   struct ini *lastpe;

   /* create a new entry to tag on to the end of the list */
   entry = malloc(sizeof(*entry));

   /* fill it up and terminate it */
   entry->section = strdup(section);
   entry->key     = strdup(key);
   entry->value   = strdup(value);
   entry->next    = NULL;

   /* find the last entry in the active list and tag this entry on the end.   */
   pe = entries;
   lastpe = NULL;
   while (pe)
   {
      lastpe = pe;
      pe = pe->next;
   }
   if (lastpe)
   {
      lastpe->next = entry;
   }
   else
   {
      entries = entry;
   }

   return;
}

