#include <windows.h>
#include <stdio.h>

#include "def.h"
#include "resource.h"
#include "params.h"
#include "Simulation.h"
#include "reports.h"
#include "util.h"

/* ************************************************************************** */
/* Some local defines for window handling                                     */
/* ************************************************************************** */
#define SIMRUN_NOT 0
#define SIMRUN_SINGLE 1
#define SIMRUN_BATCH 2

/* ************************************************************************** */
/* some globals variables because I don't yet know how to pass variables back */
/* from the call-back functions.                                              */
/* ************************************************************************** */
char str_RetAge[4] ;
char str_RetYr[5] ;
char str_RetStash[9] ;
char srcfile[MAX_PATH]="" ;

/* ************************************************************************** */
/* Window handles  (other than the main window)                               */
/* ************************************************************************** */
HWND      ghwndEdit      ; /* Text window in the main window */
HINSTANCE hInstGData     ; /* Input Data window              */
HINSTANCE hInstSaveRes   ; /* Save Results dialog box        */
HINSTANCE hInstBatch     ; /* Input Batch Data               */
HINSTANCE hInstSaveBatch ; /* Save Batch Data                */

/* ************************************************************************** */
/*  Main window name (and most dialog boxes too                               */
/* ************************************************************************** */
char MainWinName[ ] = "Monte Carlo for Windows";

/* ************************************************************************** */
/*  Declare Windows callback procedures                                       */
/* ************************************************************************** */
LRESULT CALLBACK WindowProcedure (HWND     , UINT    , WPARAM       , LPARAM       );
LRESULT CALLBACK DlgData         (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK DlgSaveRes      (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK DlgBatch        (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK DlgSBatch       (HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);

/* ************************************************************************** */
/* Other helper functions                                                     */
/* ************************************************************************** */
void OpenDialog  (HWND,char*);
void SaveAsDialog(HWND,char*);
void LoadFile    (LPSTR,HWND);

/* ************************************************************************** */
/* Fire up the main window and message dispatcher                             */
/* ************************************************************************** */
int WINAPI WinMain (HINSTANCE hThisInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR     lpszArgument,
                    int       nFunsterStil)

{
    HWND hwnd;               /* This is the handle for our window */
    MSG messages;            /* Here messages to the application are saved */
    HMENU menu;              /* Handle of the menu */
    WNDCLASSEX wincl;        /* Data structure for the windowclass */

    /* The Window structure */
    wincl.hInstance     = hThisInstance;
    wincl.lpszClassName = MainWinName;
    wincl.lpfnWndProc   = WindowProcedure;      /* This function is called by windows */
    wincl.style         = CS_DBLCLKS;           /* Catch double-clicks */
    wincl.cbSize        = sizeof (WNDCLASSEX);
    wincl.hIcon         = LoadIcon   (NULL, IDI_APPLICATION); /* default icon */
    wincl.hIconSm       = LoadIcon   (NULL, IDI_APPLICATION);
    wincl.hCursor       = LoadCursor (NULL, IDC_ARROW); /* default pointer */
    wincl.lpszMenuName  = NULL;                 /* No menu */
    wincl.cbClsExtra    = 0;                    /* No extra bytes after the window class */
    wincl.cbWndExtra    = 0;                    /* structure or the window instance */
    wincl.hbrBackground = (HBRUSH) COLOR_BACKGROUND; /* default background */

    /* Register the window class, and if it fails quit the program */
    if (!RegisterClassEx (&wincl))
        return 0;

    /* The class is registered, let's create the program*/

    hwnd = CreateWindowEx (
           WS_EX_RIGHTSCROLLBAR,/* Extended possibilites for variation */
           MainWinName,         /* Classname */
           MainWinName,         /* Title Text */
           WS_OVERLAPPEDWINDOW, /* default window */
           CW_USEDEFAULT,       /* Windows decides the position */
           CW_USEDEFAULT,       /* where the window ends up on the screen */
           544,                 /* The programs width */
           725,                 /* and height in pixels */
           HWND_DESKTOP,        /* The window is a child-window to desktop */
           NULL,                /* No menu */
           hThisInstance,       /* Program Instance handler */
           NULL                 /* No Window Creation data */
           );

    /* Make the window visible on the screen */
    ShowWindow (hwnd, nFunsterStil);

    /* Load and Set the menu tree */
    menu = LoadMenu(hThisInstance, MAKEINTRESOURCE(IDR_MENU1));
    SetMenu(hwnd, menu);

    /* Run the message loop. It will run until GetMessage() returns 0 */
    while (GetMessage (&messages, NULL, 0, 0))
    {
        /* Translate virtual-key messages into character messages */
        TranslateMessage(&messages);
        /* Send message to WindowProcedure */
        DispatchMessage(&messages);
    }

    /* The program return-value is 0 - The value that PostQuitMessage() gave */
    sims_done() ; /* free up any malloc'd data */
    return messages.wParam;
}


/* ************************************************************************** */
/*  Main Window message handler                                               */
/* ************************************************************************** */
LRESULT CALLBACK WindowProcedure (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
   HDC          hDC ; /* windowsy stuff */
   PAINTSTRUCT  Ps  ; /* windowsy stuff */

   int          error                 ; /* general error accumulator          */
   static uint8 SimRun = SIMRUN_NOT   ; /* Flag of if the simulation has run  */
   char         msg[200]              ; /* Genreal status message             */
   uint32       n,i,j,k               ; /* General counters                   */
   uint32       percent               ; /* Interim simulation percent success */

    switch (message)
    {
      case WM_CREATE:
           /* on creation, make sure params are in a known state */
           init_params() ;
           break ;
           
      case WM_PAINT :
           hDC = BeginPaint(hwnd, &Ps);
           SetTextColor(hDC, RGB(  0 , 0,  0));
           SetBkColor  (hDC, RGB(190,190,190));
           if (SimRun==SIMRUN_NOT)
           {
              TextOut(hDC, 50, 42, "Simulation Not Run Yet", 22);
           }
           else if (SimRun==SIMRUN_SINGLE)
           {
              TextOut(hDC, 50, 42, "Single Simulation Summary:", 26);

              /* Show each modes results, red for <80%, green for >=80%       */
              for (i=0;i<N_MODES;i++)
              {
                 percent = summaries[i].successes*100/params.num_sims ;
                 if (percent<80)
                 {
                    SetTextColor(hDC, RGB(255,  0,  0));
                 }
                 else
                 {
                    SetTextColor(hDC, RGB(  0,128,  0));
                 }
                 sprintf (msg,"Mode %d: %5d of %5d successes for %3d%% success rate.\n",i,summaries[i].successes,params.num_sims,percent) ;
                 TextOut(hDC, 50, 62+i*20,msg,strlen(msg));
              } ;

           }
           else /* batch run */
           {
              /* The indexing for this gets *really* creative!!! :) */
              TextOut(hDC,110, 20, "Batch Simulation Summary", 24);
              TextOut(hDC, 50, 42, "Retire When", 11);
              TextOut(hDC,185, 42, "CPI", 3);
              if (BatchDat.BothSSIs)
              {
                 TextOut(hDC,250, 42, "-SSI", 4);
                 TextOut(hDC,315, 42, "+SSI", 4);
              }
              else
              {
                 if (params.soc_sec_use)
                 {
                    TextOut(hDC,250, 42, "+SSI", 4);
                 }
                 else
                 {
                    TextOut(hDC,250, 42, "-SSI", 4);
                 }
              }


              for (n=0;n<5;n++)
              {
                 /* I want each year/age.  This is every 1th/2nd/4th entry    */
                 /* depending on if SSI and CPI variants are enabled          */
                 k=n*(nbatchs*nbatchc) ;
                 TextOut(hDC, 50, 62+n*120,batch_results[k].descr,14);
                 TextOut(hDC, 50, 82+n*120,batch_results[k].descr+16,6);

                 /* now for the CPI loop (if there is one  */
                 for (i=0;i<nbatchc;i++)
                 {
                    /* CPI index is every 1th or 2nd depending on if SSI is   */
                    /* enabled or not.                                        */
                    k=n*(nbatchs*nbatchc)+i*(nbatchs) ;
                    TextOut(hDC, 180, 82+n*120+i*60,batch_results[k].descr+29,5);

                    /* now do two columns of data if they had SSI in and out  */
                    for (j=0;j<nbatchs;j++)
                    {
                       k=n*(nbatchs*nbatchc)+i*(nbatchs)+j ;
                       sprintf(msg,"0: %3d%%",(int)batch_results[k].mode[0]) ;
                       if ((int)batch_results[k].mode[0]<80)
                       {
                          SetTextColor(hDC, RGB(255,  0,  0));
                       }
                       else
                       {
                          SetTextColor(hDC, RGB(  0,128,  0));
                       } ;
                       TextOut(hDC, 240+j*65, 62+n*120+i*60,msg,7);

                       sprintf(msg,"1: %3d%%",(int)batch_results[k].mode[1]) ;
                       if ((int)batch_results[k].mode[1]<80)
                       {
                          SetTextColor(hDC, RGB(255,  0,  0));
                       }
                       else
                       {
                          SetTextColor(hDC, RGB(  0,128,  0));
                       } ;
                       TextOut(hDC, 240+j*65, 82+n*120+i*60,msg,7);

                       sprintf(msg,"2: %3d%%",(int)batch_results[k].mode[2]) ;
                       if ((int)batch_results[k].mode[2]<80)
                       {
                          SetTextColor(hDC, RGB(255,  0,  0));
                       }
                       else
                       {
                          SetTextColor(hDC, RGB(  0,128,  0));
                       } ;
                       TextOut(hDC, 240+j*65,102+n*120+i*60,msg,7);

                    } ;
                    /* back to black */
                    SetTextColor(hDC, RGB(  0,  0,  0));
                 } ;
              } ;

              /* now draw the pretty boxes around all the numbers */
              MoveToEx(hDC, 40, 60, NULL);
              LineTo  (hDC, 40,660);
              LineTo  (hDC,365,660);
              LineTo  (hDC,365, 60);
              LineTo  (hDC, 40, 60);
              for (i=0;i<5;i++)
              {
                 MoveToEx(hDC, 40, 60+i*120, NULL);
                 LineTo  (hDC,365, 60+i*120);
                 MoveToEx(hDC,170,120+i*120, NULL);
                 LineTo  (hDC,365,120+i*120);
              }

              MoveToEx(hDC,170, 60, NULL);
              LineTo  (hDC,170,660);
              MoveToEx(hDC,230, 60, NULL);
              LineTo  (hDC,230,660);
              MoveToEx(hDC,295, 60, NULL);
              LineTo  (hDC,295,660);
              
              /* All done! */
              EndPaint(hwnd, &Ps);


//              for (i=0;i<nbatch;i++)
//              {
//                 sprintf(msg,"%s %3d%% %3d%% %3d%%",batch_results[i].descr,batch_results[i].mode[0],batch_results[i].mode[1],batch_results[i].mode[2]) ;
//                 TextOut(hDC, 50, 62+i*20,msg,strlen(msg));
//              }
           }

           EndPaint(hwnd, &Ps);
           return 0;

       case WM_COMMAND:
           /* These are all the menu tree entries.                            */
           switch( wParam )
           {
             case IDM_FILENEW:\
                  init_params() ;
                  MessageBox (hwnd, (LPSTR) "Source Parameters Reset.",
                              (LPSTR) MainWinName,
                              MB_ICONINFORMATION | MB_OK );
                  SimRun = SIMRUN_NOT ;
                  /* invalidate the whole window.            */
                  /* Not efficient, but good enough for now. */
                  InvalidateRgn(hwnd,NULL,TRUE) ;

                  return 0;

             case IDM_FILEOPEN:
                  OpenDialog(ghwndEdit,srcfile);
                  SimRun = SIMRUN_NOT ;
                  /* invalidate the whole window.            */
                  /* Not efficient, but good enough for now. */
                  InvalidateRgn(hwnd,NULL,TRUE) ;

                  return 0;

             case IDM_FILESAVE:
                  /* make sure we have a filename before saving */
                  if (srcfile[0]!=0 )
                  {
                     save_params(srcfile) ;
                     return 0 ;
                  }
                  /* else fall into Save As */

             case IDM_FILESAVEAS:
                  SaveAsDialog(ghwndEdit,srcfile);
                  return 0;

             case IDM_INPUTDATA:
                  DialogBox(
                  hInstGData,
                  MAKEINTRESOURCE(IDD_DIALOG_INPUT),
	               hwnd,
                  (DLGPROC)DlgData);

                  SimRun = SIMRUN_NOT ;
                  /* invalidate the whole window.            */
                  /* Not efficient, but good enough for now. */
                  InvalidateRgn(hwnd,NULL,TRUE) ;

                 /* if any errors, say so and tell 'em to try again */
                 if (Perror[0])
                 {
                     MessageBox (hwnd, (LPSTR) "Errors in input data.  Fields in error are highlighted in red.",
                                 (LPSTR) MainWinName,
                                 MB_ICONINFORMATION | MB_OK );
                 }

                 /* if logs or debug is on and # of sims is >10, warn them! */
                 if ((params.logs||params.debug)&&(params.num_sims>10))
                 {
                     MessageBox (hwnd, (LPSTR) "WARNING: Enabling logs and/or debug information\nwith more than 10 simulations will generate a\nvery large log file",
                                 (LPSTR) MainWinName,
                                 MB_ICONINFORMATION | MB_OK );
                 }

                  return 0;

             case IDM_GO:
                  error = run_sims() ;
                  if (error)
                  {
                     MessageBox (hwnd, (LPSTR) "Simulation Errors!",
                                 (LPSTR) MainWinName,
                                 MB_ICONINFORMATION | MB_OK );
                  }
                  else
                  {
                     SimRun = SIMRUN_SINGLE ;
                     /* invalidate the whole window so the status gets updated */
                     /* Not efficient, but good enough for now.                */
                     InvalidateRgn(hwnd,NULL,TRUE) ;
                  }
                  return 0;

             case IDM_BATCH:
                  DialogBox(
                  hInstBatch,
                  MAKEINTRESOURCE(IDD_BATCH),
	               hwnd,
                  (DLGPROC)DlgBatch);

                  SimRun = SIMRUN_NOT ;
                  /* invalidate the whole window.            */
                  /* Not efficient, but good enough for now. */
                  InvalidateRgn(hwnd,NULL,TRUE) ;

                 /* if any errors, say so and tell 'em to try again */
                 if (Berror[0])
                 {
                     MessageBox (hwnd, (LPSTR) "Errors in input data.  Fields in error are highlighted in red.",
                                 (LPSTR) MainWinName,
                                 MB_ICONINFORMATION | MB_OK );
                 }

                  return (0) ;

             case IDM_RUN_BATCH:
                  error = run_batch() ;
                  if (error)
                  {
                     MessageBox (hwnd, (LPSTR) "Batch Run Simulation Errors!",
                                 (LPSTR) MainWinName,
                                 MB_ICONINFORMATION | MB_OK );
                  }
                  else
                  {
                     SimRun = SIMRUN_BATCH ;
                     /* invalidate the whole window so the status gets updated */
                     /* Not efficient, but good enough for now.                */
                     InvalidateRgn(hwnd,NULL,TRUE) ;
                  }
                  return 0;



             case IDM_SAVE_RESULTS:
                  if(srcfile[0]==0) /* no source file yet */
                  {
                     MessageBox (hwnd, (LPSTR) "Please load/save source data first.",
                             (LPSTR) MainWinName,
                             MB_ICONINFORMATION | MB_OK );
                  }
                  else if(SimRun!=SIMRUN_SINGLE) /* haven't run the simulation yet */
                  {
                     MessageBox (hwnd, (LPSTR) "Please run a single simulation first.",
                             (LPSTR) MainWinName,
                             MB_ICONINFORMATION | MB_OK );
                  }
                  else
                  {
                     DialogBox(
                        hInstSaveRes,
                        MAKEINTRESOURCE(IDD_SAVE_RESULTS),
	                     hwnd,
                        (DLGPROC)DlgSaveRes);
                  }
                  return 0;

             case IDM__BATCH_SAVE:
                  if(srcfile[0]==0) /* no source file yet */
                  {
                     MessageBox (hwnd, (LPSTR) "Please load/save source data first.",
                             (LPSTR) MainWinName,
                             MB_ICONINFORMATION | MB_OK );
                  }
                  else if(SimRun!=SIMRUN_BATCH) /* haven't run the simulation yet */
                  {
                     MessageBox (hwnd, (LPSTR) "Please run a batch simulation first.",
                             (LPSTR) MainWinName,
                             MB_ICONINFORMATION | MB_OK );
                  }
                  else
                  {
                     DialogBox(
                        hInstSaveBatch,
                        MAKEINTRESOURCE(IDD_SAVE_RESULTS),
	                     hwnd,
                        (DLGPROC)DlgSBatch);
                  }
                  return 0;

             case IDM_HELPCONTENTS:
                  ShellExecute(hwnd,"open","WinMonte Help.pdf",NULL,NULL,SW_SHOWNORMAL);
                  return 0;

             case IDM_FILEEXIT:
                  SendMessage( hwnd, WM_CLOSE, 0, 0L );
                  return 0;

             case IDM_HELPABOUT:
                  MessageBox (NULL, "Wills Windows-based Monte Carlo Simulator\n Rev 1.00", "About Monte Carlo" ,MB_OK|MB_ICONINFORMATION);
                  return 0;

             default:
                  MessageBox (hwnd, (LPSTR) "Oops, unknown menu function!",
                              (LPSTR) MainWinName,
                              MB_ICONINFORMATION | MB_OK );
                  return 0;

           }
           break;

        case WM_CLOSE:
           DestroyWindow( hwnd );
           return 0;

        case WM_DESTROY:
            PostQuitMessage (0);       /* send a WM_QUIT to the message queue */
            break;
        default:                      /* for messages that we don't deal with */
            return DefWindowProc (hwnd, message, wParam, lParam);
    }

    return 0;
}

/* ************************************************************************** */
/* The File Open dialog box                                                   */
/* ************************************************************************** */
void OpenDialog(HWND ghwndEdit,char* srcfile)
{
   OPENFILENAME ofn;
   TCHAR szFile[MAX_PATH];
   char error ;

 
   /* Set up the parameters for file to open */
   ZeroMemory(&ofn, sizeof(ofn));
   ofn.lStructSize     = sizeof(ofn);
   ofn.lpstrFile       = szFile;
   ofn.lpstrFile[0]    = '\0';
   ofn.hwndOwner       = ghwndEdit;
   ofn.nMaxFile        = sizeof(szFile);
   ofn.lpstrFilter     = TEXT("Data Files (*.ini)\0*.ini\0");
   ofn.nFilterIndex    = 1;
   ofn.lpstrInitialDir = NULL;
   ofn.lpstrFileTitle  = NULL;
   ofn.Flags           = OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;
  
   /* open the file and if success, load the data */
   if(GetOpenFileName(&ofn))
   {
      /* load the data and see if errors */
      error=read_params(ofn.lpstrFile,FALSE) ;
      strncpy(srcfile,ofn.lpstrFile,MAX_PATH) ;

      /* if any errors in the input file, say so and tell 'em to check */
      if (Perror[0])
      {
          MessageBox (ghwndEdit, (LPSTR) "Errors data file.  Please check source data. Parameters in error are highlighted in red.",
                      (LPSTR) MainWinName,
                      MB_ICONINFORMATION | MB_OK );
      }

      /* all done */
   } ;
} ;

/* ************************************************************************** */
/* The Save As dialog box                                                     */
/* ************************************************************************** */
void SaveAsDialog(HWND ghwndEdit,char* srcfile)
{
   OPENFILENAME ofn;
   TCHAR szFile[MAX_PATH];

   /* Set up the parameters for file to open */
   ZeroMemory(&ofn, sizeof(ofn));
   ofn.lStructSize     = sizeof(ofn);
   ofn.lpstrFile       = szFile;
   ofn.hwndOwner       = ghwndEdit;
   ofn.nMaxFile        = sizeof(szFile);
   ofn.lpstrFilter     = TEXT("Data Files (*.ini)\0*.ini\0");
   ofn.nFilterIndex    = 1;
   ofn.lpstrInitialDir = NULL;
   ofn.lpstrFileTitle  = NULL;
   ofn.Flags           = OFN_PATHMUSTEXIST;

   /* Defult to the opened file (it it existed) */
   strncpy(ofn.lpstrFile,srcfile,MAX_PATH) ;

   /* open the file and if success, load the data */
   if(GetOpenFileName(&ofn))
   {
      /* load the data and see if errors */
      save_params(ofn.lpstrFile) ;
      /* Copy the filename back for saving */
      strncpy(srcfile,ofn.lpstrFile,MAX_PATH) ;
      /* all done */
   } ;
} ;

/* ************************************************************************** */
/* Callback function for the Input Data dialog box                            */
/* For reviewing and editing the simulation input data                        */
/* ************************************************************************** */
LRESULT CALLBACK DlgData(HWND hWndDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{
   char text[20] ; /* max string for conversions */
   uint8 i ;

   switch(Msg)
   {
   case WM_INITDIALOG:
        sprintf(text,"%d"   ,params.start_age                      ) ; SetDlgItemText(hWndDlg,IDC_RETIRE_AGE  ,text) ;
        sprintf(text,"%d"   ,params.start_year                     ) ; SetDlgItemText(hWndDlg,IDC_RETIRE_YEAR ,text) ;
        itoc(text,(int32)params.start_amount,10,'$',TRUE           ) ; SetDlgItemText(hWndDlg,IDC_STASH       ,text) ;
        sprintf(text,"%d"   ,params.end_age                        ) ; SetDlgItemText(hWndDlg,IDC_END_AGE     ,text) ;
        sprintf(text,"%8.3f",params.mkt_return_avg                 ) ; SetDlgItemText(hWndDlg,IDC_STAGE1_INT  ,text) ;
        sprintf(text,"%8.3f",params.mkt_return_range               ) ; SetDlgItemText(hWndDlg,IDC_STAGE1_RANGE,text) ;

        sprintf(text,"%d"   ,params.age_ror_phase2                 ) ; SetDlgItemText(hWndDlg,IDC_STAGE2_AGE  ,text) ;
        sprintf(text,"%8.3f",params.mkt_return_avg2                ) ; SetDlgItemText(hWndDlg,IDC_STAGE2_INT  ,text) ;
        sprintf(text,"%8.3f",params.mkt_return_range2              ) ; SetDlgItemText(hWndDlg,IDC_STAGE2_RANGE,text) ;

        sprintf(text,"%d"   ,params.age_ror_phase3                 ) ; SetDlgItemText(hWndDlg,IDC_STAGE3_AGE  ,text) ;
        sprintf(text,"%8.3f",params.mkt_return_avg3                ) ; SetDlgItemText(hWndDlg,IDC_STAGE3_INT  ,text) ;
        sprintf(text,"%8.3f",params.mkt_return_range3              ) ; SetDlgItemText(hWndDlg,IDC_STAGE3_RANGE,text) ;

        sprintf(text,"%s"   ,params.cat1_expense_name              ) ; SetDlgItemText(hWndDlg,IDC_GROUP1_NAME ,text) ;
        itoc(text,(int32)params.cat1_expense_amount_min,10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_GROUP1_MIN  ,text) ;
        itoc(text,(int32)params.cat1_expense_amount_max,10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_GROUP1_MAX  ,text) ;
        sprintf(text,"%8.3f",params.cat1_expense_inflation         ) ; SetDlgItemText(hWndDlg,IDC_GROUP1_INF  ,text) ;

        sprintf(text,"%s"   ,params.cat2_expense_name              ) ; SetDlgItemText(hWndDlg,IDC_GROUP2_NAME ,text) ;
        itoc(text,(int32)params.cat2_expense_amount_min,10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_GROUP2_MIN  ,text) ;
        itoc(text,(int32)params.cat2_expense_amount_max,10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_GROUP2_MAX  ,text) ;
        sprintf(text,"%8.3f",params.cat2_expense_inflation         ) ; SetDlgItemText(hWndDlg,IDC_GROUP2_INF  ,text) ;

        sprintf(text,"%s"   ,params.cat3_expense_name              ) ; SetDlgItemText(hWndDlg,IDC_GROUP3_NAME ,text) ;
        itoc(text,(int32)params.cat3_expense_amount_min,10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_GROUP3_MIN  ,text) ;
        itoc(text,(int32)params.cat3_expense_amount_max,10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_GROUP3_MAX  ,text) ;
        sprintf(text,"%8.3f",params.cat3_expense_inflation         ) ; SetDlgItemText(hWndDlg,IDC_GROUP3_INF  ,text) ;

        sprintf(text,"%d"   ,params.soc_sec_age_start              ) ; SetDlgItemText(hWndDlg,IDC_SOCSEC_START,text) ;
        itoc(text,(int32)params.soc_sec_amt,10,'$',TRUE            ) ; SetDlgItemText(hWndDlg,IDC_SOCSEC_AMT  ,text) ;
        sprintf(text,"%8.3f",params.soc_sec_inflation              ) ; SetDlgItemText(hWndDlg,IDC_SOCSEC_COLA ,text) ;

        sprintf(text,"%d"   ,params.num_sims                       ) ; SetDlgItemText(hWndDlg,IDC_NUMSIMS     ,text) ;

        if (params.soc_sec_use   ) SendDlgItemMessage(hWndDlg,IDC_SOCSEC_USE     ,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;
        if (params.logs          ) SendDlgItemMessage(hWndDlg,IDC_CHECK_LOGS     ,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;
        if (params.debug         ) SendDlgItemMessage(hWndDlg,IDC_CHECK_DEBUG    ,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;

        return TRUE;

   /* my attempt at setting dynamic colors if fields have errors */
   case WM_CTLCOLOREDIT:
        {
           HDC hdcStatic = (HDC) wParam;
           int handle ;

           /* Default color unless overridden (black) */
           SetTextColor(hdcStatic, RGB(  0,  0,  0)) ;
           SetBkColor  (hdcStatic, RGB(236,236,236)) ;

           handle = GetDlgCtrlID((HWND)lParam) ;
           switch (handle)
           {
              case IDC_STASH       : if (Perror[ 1]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_RETIRE_YEAR : if (Perror[ 2]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_RETIRE_AGE  : if (Perror[ 3]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_END_AGE     : if (Perror[ 4]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              case IDC_STAGE1_INT  : if (Perror[ 5]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_STAGE1_RANGE: if (Perror[ 6]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              case IDC_STAGE2_AGE  : if (Perror[ 7]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_STAGE2_INT  : if (Perror[ 8]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_STAGE2_RANGE: if (Perror[ 9]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              case IDC_STAGE3_AGE  : if (Perror[10]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_STAGE3_INT  : if (Perror[11]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_STAGE3_RANGE: if (Perror[12]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              case IDC_GROUP1_MIN  : if (Perror[14]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_GROUP1_MAX  : if (Perror[15]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_GROUP1_INF  : if (Perror[16]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              case IDC_GROUP2_MIN  : if (Perror[18]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_GROUP2_MAX  : if (Perror[19]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_GROUP2_INF  : if (Perror[20]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              case IDC_GROUP3_MIN  : if (Perror[22]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_GROUP3_MAX  : if (Perror[23]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_GROUP3_INF  : if (Perror[24]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              case IDC_SOCSEC_START: if (Perror[25]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_SOCSEC_AMT  : if (Perror[26]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_SOCSEC_COLA : if (Perror[27]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              case IDC_NUMSIMS     : if (Perror[29]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              /* all other fiels, leave black */
              default :                         SetTextColor(hdcStatic, RGB(  0,0,0)) ; break ;
           } ;
           return (INT_PTR)GetSysColorBrush(COLOR_WINDOW) ;
        }
        break;

   case WM_COMMAND:
        switch(wParam)
        {
        case IDC_OK:
             /* They hit OK,store all the params */
             GetDlgItemText(hWndDlg,IDC_STASH       ,text,20) ; Perror[ 1]=ctof_e(&params.start_amount            , text) ;
             GetDlgItemText(hWndDlg,IDC_RETIRE_YEAR ,text,20) ; Perror[ 2]=atoi_e(&params.start_year              , text) ;
             GetDlgItemText(hWndDlg,IDC_RETIRE_AGE  ,text,20) ; Perror[ 3]=atoi_e(&params.start_age               , text) ;
             GetDlgItemText(hWndDlg,IDC_END_AGE     ,text,20) ; Perror[ 4]=atoi_e(&params.end_age                 , text) ;

             GetDlgItemText(hWndDlg,IDC_STAGE1_INT  ,text,20) ; Perror[ 5]=atof_e(&params.mkt_return_avg          , text) ;
             GetDlgItemText(hWndDlg,IDC_STAGE1_RANGE,text,20) ; Perror[ 6]=atof_e(&params.mkt_return_range        , text) ;

             GetDlgItemText(hWndDlg,IDC_STAGE2_AGE  ,text,20) ; Perror[ 7]=atoi_e(&params.age_ror_phase2          , text) ;
             GetDlgItemText(hWndDlg,IDC_STAGE2_INT  ,text,20) ; Perror[ 8]=atof_e(&params.mkt_return_avg2         , text) ;
             GetDlgItemText(hWndDlg,IDC_STAGE2_RANGE,text,20) ; Perror[ 9]=atof_e(&params.mkt_return_range2       , text) ;
                                                         
             GetDlgItemText(hWndDlg,IDC_STAGE3_AGE  ,text,20) ; Perror[10]=atoi_e(&params.age_ror_phase3          , text) ;
             GetDlgItemText(hWndDlg,IDC_STAGE3_INT  ,text,20) ; Perror[11]=atof_e(&params.mkt_return_avg3         , text) ;
             GetDlgItemText(hWndDlg,IDC_STAGE3_RANGE,text,20) ; Perror[12]=atof_e(&params.mkt_return_range3       , text) ;
                                                         
             GetDlgItemText(hWndDlg,IDC_GROUP1_NAME ,params.cat1_expense_name,20) ;
             GetDlgItemText(hWndDlg,IDC_GROUP1_MIN  ,text,20) ; Perror[14]=ctof_e(&params.cat1_expense_amount_min , text) ;
             GetDlgItemText(hWndDlg,IDC_GROUP1_MAX  ,text,20) ; Perror[15]=ctof_e(&params.cat1_expense_amount_max , text) ;
             GetDlgItemText(hWndDlg,IDC_GROUP1_INF  ,text,20) ; Perror[16]=atof_e(&params.cat1_expense_inflation  , text) ;
                                                         
             GetDlgItemText(hWndDlg,IDC_GROUP2_NAME ,params.cat2_expense_name,20) ;
             GetDlgItemText(hWndDlg,IDC_GROUP2_MIN  ,text,20) ; Perror[18]=ctof_e(&params.cat2_expense_amount_min , text) ;
             GetDlgItemText(hWndDlg,IDC_GROUP2_MAX  ,text,20) ; Perror[19]=ctof_e(&params.cat2_expense_amount_max , text) ;
             GetDlgItemText(hWndDlg,IDC_GROUP2_INF  ,text,20) ; Perror[20]=atof_e(&params.cat2_expense_inflation  , text) ;
                                                         
             GetDlgItemText(hWndDlg,IDC_GROUP3_NAME ,params.cat3_expense_name,20) ;
             GetDlgItemText(hWndDlg,IDC_GROUP3_MIN  ,text,20) ; Perror[22]=ctof_e(&params.cat3_expense_amount_min , text) ;
             GetDlgItemText(hWndDlg,IDC_GROUP3_MAX  ,text,20) ; Perror[23]=ctof_e(&params.cat3_expense_amount_max , text) ;
             GetDlgItemText(hWndDlg,IDC_GROUP3_INF  ,text,20) ; Perror[24]=atof_e(&params.cat3_expense_inflation  , text) ;
                                                         
             GetDlgItemText(hWndDlg,IDC_SOCSEC_START,text,20) ; Perror[25]=atoi_e(&params.soc_sec_age_start       , text) ;
             GetDlgItemText(hWndDlg,IDC_SOCSEC_AMT  ,text,20) ; Perror[26]=ctof_e(&params.soc_sec_amt,text) ;
             GetDlgItemText(hWndDlg,IDC_SOCSEC_COLA ,text,20) ; Perror[27]=atof_e(&params.soc_sec_inflation       , text) ;
                                                         
             GetDlgItemText(hWndDlg,IDC_NUMSIMS     ,text,20) ; Perror[29]=atoi_e(&params.num_sims                , text) ;
      
             params.soc_sec_use    = (SendDlgItemMessage(hWndDlg,IDC_SOCSEC_USE     ,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;
             params.logs           = (SendDlgItemMessage(hWndDlg,IDC_CHECK_LOGS     ,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;
             params.debug          = (SendDlgItemMessage(hWndDlg,IDC_CHECK_DEBUG    ,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;

             /* The parameters that can't have errors */
             Perror[ 0]=FALSE; /* Summary flag       */
             Perror[13]=FALSE; /* Cat1 Name          */
             Perror[17]=FALSE; /* Cat2 Name          */
             Perror[21]=FALSE; /* Cat3 Name          */
             Perror[28]=FALSE; /* SocSec check       */
             Perror[30]=FALSE; /* Logs check         */
             Perror[31]=FALSE; /* Detail check       */
             Perror[32]=FALSE; /* Histogram check    */
             Perror[33]=FALSE; /* Debug check        */
             Perror[34]=FALSE; /* Summary Filename   */
             Perror[35]=FALSE; /* Histogram Filename */

             /* Use Perror[0] as a summary of all errors */
             for(i=1;i<36;i++) Perror[0]|=Perror[i] ;

             EndDialog(hWndDlg, 0);
             return TRUE;

       case IDC_CANCEL:
             EndDialog(hWndDlg, 0);
             return TRUE;
       }
       break;

    case WM_CLOSE:
         EndDialog(hWndDlg, 0);
         return TRUE;
   }

   return FALSE;
}


/* ************************************************************************** */
/* Callback function for the Save Results dialog box                          */
/* For reviewing and editing the simulation input data                        */
/* ************************************************************************** */
LRESULT CALLBACK DlgSaveRes(HWND hWndDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{
   static char   srcpath[MAX_PATH] ;
   char   dstfile[MAX_PATH] ;
   FILE*  fout              ;
   uint32 i                 ;

   switch(Msg)
   {
   case WM_INITDIALOG:
        /* get just the path from the source filename.  The source filename   */
        /* must exist before this routine can be called.                      */
        strncpy(srcpath,srcfile,MAX_PATH) ;
        for (i=strlen(srcpath);i>0;i--)
        {
           if (srcpath[i]=='\\') break ;
        }
        /* i will be at the last \ or beginning of the file, terminate it.    */
        srcpath[i]=0 ;

        SetDlgItemText(hWndDlg,IDC_PATH  ,srcpath) ;
        SetDlgItemText(hWndDlg,IDC_SUMMARY_FILE  ,params.SummaryFile  ) ;
        SetDlgItemText(hWndDlg,IDC_HISTOGRAM_FILE,params.HistogramFile) ;

        if (params.detail_summary) SendDlgItemMessage(hWndDlg,IDC_SUMMARY_DETAIL ,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;
        if (params.histogram     ) SendDlgItemMessage(hWndDlg,IDC_HISTOGRAM      ,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;

        return TRUE;

   case WM_COMMAND:
        switch(wParam)
        {
        case IDC_OK:
             /* They hit OK,store all the params */

             GetDlgItemText(hWndDlg,IDC_SUMMARY_FILE ,params.SummaryFile,MAX_PATH) ;
             GetDlgItemText(hWndDlg,IDC_HISTOGRAM_FILE ,params.HistogramFile,MAX_PATH) ;
             params.detail_summary = (SendDlgItemMessage(hWndDlg,IDC_SUMMARY_DETAIL ,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;
             params.histogram      = (SendDlgItemMessage(hWndDlg,IDC_HISTOGRAM      ,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;

             /* Got the results, now write out the results */
             strncpy(dstfile,srcpath,MAX_PATH);
             strcat(dstfile,"\\") ;
             strcat(dstfile,params.SummaryFile) ;

             if ((fout=fopen(dstfile,"w")) == NULL)
             {
                perror("report file error:");
                exit(1) ;
             } ;

             /* now just create the histogram filename, it will only be opened */
             /* if a histogram is requested.                                   */
             strncpy(dstfile,srcpath,MAX_PATH);
             strcat(dstfile,"\\") ;
             strcat(dstfile,params.HistogramFile) ;

             /* get the data and report it */
             write_params(fout) ;
             report_percentile(results,fout)          ;
             report_histogram(summaries,results,fout,dstfile) ;
             report_summaries(summaries,results,fout) ;
             fclose(fout) ;

             EndDialog(hWndDlg, 0);
             return TRUE;

       case IDC_CANCEL:
             EndDialog(hWndDlg, 0);
             return TRUE;
       }
       break;
    case WM_CLOSE:
         EndDialog(hWndDlg, 0);
         return TRUE;

   }

   return FALSE;
}

/* ************************************************************************** */
/* Callback function for entering the Batch data inputs                       */
/* ************************************************************************** */
LRESULT CALLBACK DlgBatch(HWND hWndDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{
   char   tmptxt[20] ;
   int i ;

   switch(Msg)
   {
   /* my attempt at setting dynamic colors if fields have errors */
   case WM_CTLCOLOREDIT:
        {
           HDC hdcStatic = (HDC) wParam;
           int handle ;

           /* Default color unless overridden (black) */
           SetTextColor(hdcStatic, RGB(  0,  0,  0)) ;
           SetBkColor  (hdcStatic, RGB(236,236,236)) ;

           handle = GetDlgCtrlID((HWND)lParam) ;
           switch (handle)
           {
              case IDC_STASH1 : if (Berror[ 1]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_STASH2 : if (Berror[ 2]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_STASH3 : if (Berror[ 3]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_STASH4 : if (Berror[ 4]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_STASH5 : if (Berror[ 5]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;
              case IDC_ALTCPI : if (Berror[ 6]) SetTextColor(hdcStatic, RGB(255,0,0)) ; break ;

              /* all other fiels, leave black */
              default :                         SetTextColor(hdcStatic, RGB(  0,0,0)) ; break ;
           } ;
           return (INT_PTR)GetSysColorBrush(COLOR_WINDOW) ;
        }
        break;

   case WM_INITDIALOG:

        /* Fill in the ages for the five years to simulate                         */
        /* I start with the age of the base simulation and go four years past that */
        sprintf(tmptxt,"%2d",params.start_age-2 ) ; SetDlgItemText(hWndDlg,IDC_RAGE1 ,tmptxt) ;
        sprintf(tmptxt,"%2d",params.start_age-1 ) ; SetDlgItemText(hWndDlg,IDC_RAGE2 ,tmptxt) ;
        sprintf(tmptxt,"%2d",params.start_age   ) ; SetDlgItemText(hWndDlg,IDC_RAGE3 ,tmptxt) ;
        sprintf(tmptxt,"%2d",params.start_age+1 ) ; SetDlgItemText(hWndDlg,IDC_RAGE4 ,tmptxt) ;
        sprintf(tmptxt,"%2d",params.start_age+2 ) ; SetDlgItemText(hWndDlg,IDC_RAGE5 ,tmptxt) ;

        /* Fill in the years for the five years to simulate                         */
        /* I start with the year of the base simulation and go four years past that */
        sprintf(tmptxt,"%4d",params.start_year-2) ; SetDlgItemText(hWndDlg,IDC_RYEAR1,tmptxt) ;
        sprintf(tmptxt,"%4d",params.start_year-1) ; SetDlgItemText(hWndDlg,IDC_RYEAR2,tmptxt) ;
        sprintf(tmptxt,"%4d",params.start_year  ) ; SetDlgItemText(hWndDlg,IDC_RYEAR3,tmptxt) ;
        sprintf(tmptxt,"%4d",params.start_year+1) ; SetDlgItemText(hWndDlg,IDC_RYEAR4,tmptxt) ;
        sprintf(tmptxt,"%4d",params.start_year+2) ; SetDlgItemText(hWndDlg,IDC_RYEAR5,tmptxt) ;

        /* the initial stash for the first age comes from the base simulation data */
        /* Rest carry forward without mods                                         */
        BatchDat.BStash[2]=params.start_amount ;
        itoc(tmptxt,(int32)BatchDat.BStash[0],10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_STASH1,tmptxt) ;
        itoc(tmptxt,(int32)BatchDat.BStash[1],10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_STASH2,tmptxt) ;
        itoc(tmptxt,(int32)BatchDat.BStash[2],10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_STASH3,tmptxt) ;
        itoc(tmptxt,(int32)BatchDat.BStash[3],10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_STASH4,tmptxt) ;
        itoc(tmptxt,(int32)BatchDat.BStash[4],10,'$',TRUE) ; SetDlgItemText(hWndDlg,IDC_STASH5,tmptxt) ;

        sprintf(tmptxt,"%8.3f",BatchDat.AltCPI) ; SetDlgItemText(hWndDlg,IDC_ALTCPI  ,tmptxt) ;

        if (BatchDat.UseAltCPI1) SendDlgItemMessage(hWndDlg,IDC_ALTCPI_C1,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;
        if (BatchDat.UseAltCPI2) SendDlgItemMessage(hWndDlg,IDC_ALTCPI_C2,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;
        if (BatchDat.UseAltCPI3) SendDlgItemMessage(hWndDlg,IDC_ALTCPI_C3,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;
        if (BatchDat.BothSSIs  ) SendDlgItemMessage(hWndDlg,IDC_SSI_BOTH ,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;

        return TRUE;

   case WM_COMMAND:
        switch(wParam)
        {
        case IDC_OK:
             /* They hit OK,store all the params */
             GetDlgItemText(hWndDlg,IDC_STASH1,tmptxt,20) ; Berror[1]=ctof_e(&BatchDat.BStash[0],tmptxt) ;
             GetDlgItemText(hWndDlg,IDC_STASH2,tmptxt,20) ; Berror[2]=ctof_e(&BatchDat.BStash[1],tmptxt) ;
             GetDlgItemText(hWndDlg,IDC_STASH3,tmptxt,20) ; Berror[3]=ctof_e(&BatchDat.BStash[2],tmptxt) ;
             GetDlgItemText(hWndDlg,IDC_STASH4,tmptxt,20) ; Berror[4]=ctof_e(&BatchDat.BStash[3],tmptxt) ;
             GetDlgItemText(hWndDlg,IDC_STASH5,tmptxt,20) ; Berror[5]=ctof_e(&BatchDat.BStash[4],tmptxt) ;

             GetDlgItemText(hWndDlg,IDC_ALTCPI,tmptxt,20) ; Berror[6]=atof_e(&BatchDat.AltCPI   ,tmptxt) ;

             BatchDat.UseAltCPI1 = (SendDlgItemMessage(hWndDlg,IDC_ALTCPI_C1,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;
             BatchDat.UseAltCPI2 = (SendDlgItemMessage(hWndDlg,IDC_ALTCPI_C2,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;
             BatchDat.UseAltCPI3 = (SendDlgItemMessage(hWndDlg,IDC_ALTCPI_C3,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;
             BatchDat.BothSSIs   = (SendDlgItemMessage(hWndDlg,IDC_SSI_BOTH ,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;

             /* The parameters that can't have errors */
             Berror[ 0]=FALSE; /* Summary flag        */
             Berror[ 7]=FALSE; /* AltCPI 1            */
             Berror[ 8]=FALSE; /* AltCPI 2            */
             Berror[ 9]=FALSE; /* AltCPI 3            */
             Berror[10]=FALSE; /* SocSecs             */

             /* Use Berror[0] as a summary of all errors */
             for(i=1;i<6;i++) Berror[0]|=Berror[i] ;

             EndDialog(hWndDlg, 0);
             return TRUE;

       case IDC_CANCEL:
             EndDialog(hWndDlg, 0);
             return TRUE;
       }
       break;
    case WM_CLOSE:
         EndDialog(hWndDlg, 0);
         return TRUE;

   }

   return FALSE;
}

/* ************************************************************************** */
/* Callback function for the Save Batch Results dialog box                    */
/* ************************************************************************** */
LRESULT CALLBACK DlgSBatch(HWND hWndDlg, UINT Msg, WPARAM wParam, LPARAM lParam)
{
   static char   srcpath[MAX_PATH] ;
   char   dstfile[MAX_PATH] ;
   FILE*  fout              ;
   uint32 i                 ;

   switch(Msg)
   {
   case WM_INITDIALOG:
        /* get just the path from the source filename.  The source filename   */
        /* must exist before this routine can be called.                      */
        strncpy(srcpath,srcfile,MAX_PATH) ;
        for (i=strlen(srcpath);i>0;i--)
        {
           if (srcpath[i]=='\\') break ;
        }
        /* i will be at the last \ or beginning of the file, terminate it.    */
        srcpath[i]=0 ;

        SetDlgItemText(hWndDlg,IDC_PATH  ,srcpath) ;
        SetDlgItemText(hWndDlg,IDC_SUMMARY_FILE  ,params.SummaryFile  ) ;
        SetDlgItemText(hWndDlg,IDC_HISTOGRAM_FILE,params.HistogramFile) ;

        if (params.detail_summary) SendDlgItemMessage(hWndDlg,IDC_SUMMARY_DETAIL ,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;
        if (params.histogram     ) SendDlgItemMessage(hWndDlg,IDC_HISTOGRAM      ,BM_SETCHECK,BST_CHECKED,(LPARAM)NULL) ;

        return TRUE;

   case WM_COMMAND:
        switch(wParam)
        {
        case IDC_OK:
             /* They hit OK,store all the params */

             GetDlgItemText(hWndDlg,IDC_SUMMARY_FILE ,params.SummaryFile,MAX_PATH) ;
             GetDlgItemText(hWndDlg,IDC_HISTOGRAM_FILE ,params.HistogramFile,MAX_PATH) ;
             params.detail_summary = (SendDlgItemMessage(hWndDlg,IDC_SUMMARY_DETAIL ,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;
             params.histogram      = (SendDlgItemMessage(hWndDlg,IDC_HISTOGRAM      ,BM_GETCHECK,(WPARAM)NULL,(LPARAM)NULL)==BST_CHECKED) ;

             /* Got the results, now write out the results */
             strncpy(dstfile,srcpath,MAX_PATH);
             strcat(dstfile,"\\") ;
             strcat(dstfile,params.SummaryFile) ;

             if ((fout=fopen(dstfile,"w")) == NULL)
             {
                perror("report file error:");
                exit(1) ;
             } ;

             /* now just create the histogram filename, it will only be opened */
             /* if a histogram is requested.                                   */
             strncpy(dstfile,srcpath,MAX_PATH);
             strcat(dstfile,"\\") ;
             strcat(dstfile,params.HistogramFile) ;

             /* get the data and report it */
             write_params(fout) ;
             report_percentile(results,fout)          ;
             report_histogram(summaries,results,fout,dstfile) ;
             report_summaries(summaries,results,fout) ;
             fclose(fout) ;

             EndDialog(hWndDlg, 0);
             return TRUE;

       case IDC_CANCEL:
             EndDialog(hWndDlg, 0);
             return TRUE;
       }
       break;
    case WM_CLOSE:
         EndDialog(hWndDlg, 0);
         return TRUE;

   }

   return FALSE;
}



//---------------------------------------------------------------------------
