
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "def.h"
#include "util.h"

/****************************************************************************
This file contains some general purpose utilities.
trimwhite - Trims any leading and trailing spaces from a string
dequote   - Removes any quotes from a string? (haven't messed with it)
itoc      - Converts integer to textcurrency format
ctoi_e    - Converts text currency format to integer
ctof_e    - Converts text currency format to float (no decimals [yet])
atoi_e    - Converts text integer to integer
atof_e    - Converts text float to float


***************************************************************************/

void
trimwhite(char *buf)
{
        int i;

        /* trim white on left */
        for (i=0; buf[i] && isspace((int)buf[i]); ++i) {
                /* empty */
        }

        if (i)
                strcpy(buf, buf+i);

        /* trim white on right */
        for (i=strlen(buf)-1; (i>=0) && isspace((int)buf[i]); --i) {
                buf[i] = '\0';
        }
}

/***************************************************************************/

void
dequote(char *string)
{
	int i, j;
	char *tmpbuf;

	tmpbuf = malloc(strlen(string)+1);

	for (i=j=0; string[i]; ++i)
	{
		if (string[i] != '"')
		{
			tmpbuf[j++] = string[i];
		}
	}
	tmpbuf[j] = '\0';

	strcpy(string, tmpbuf);
}

/****************************************************************************
* Routne   : itoc
* Gazintas : num = number to convert
*          : ndig = number of digits to convert
*          : sym = optional leading symbol
*          : comma flag to add thousands separators (commas)
* IOs      : str = pointer to string to put result
* Returns  : None
* Globals  : None
*
* This routine converts a 32-bit signed integer into currency format.
* The number is left-justified up to ndig digits.  If the number is larger
* than the specfied digits, all asterisks will be returned.  Optional
* commas are inserted every three digits and an optional currency symbol
* can be added too.  Negative numbers are OK too.
*
* The max string is 16 characters. For lesser sizes it is ndig plus the optional
* commas, currency, and negative sign (plus one for terminating null)
*
****************************************************************************/
void itoc(char* str,int32 num,uint8 ndig,char sym,uint8 comma)
{
   uint8 i,s,len ;
   int8 sign=1 ;
   char* loc ;

   /* if negative, remember that and convert positively */
   if (num<0)
   {
      sign = -1 ;
      num = -num ;
   } ;

   /* for a uint32, max is 10 digits, so peg if >10 */
   if (ndig>10) ndig=10 ;

   /* independent pointer to start of string to build it up */
   loc=str ;

   /* Actually going to build the string backwards first starting with LSD. */
   for(i=0;i<ndig;i++)
   {
      /* plunk in a digit */
      *loc++=(num%10)+'0' ;

      /* Modulo 10 for next digit */
      num /= 10 ;

      /* if we get to zero before last requested digit, break */
      if (num==0) break ;

      /* if a 1000's and comma enabled, add a comma */
      if(comma&&(i%3==2)) *loc++=',' ;
   } ;

   /* if we exited with anything other than zero, number didn't fit */
   /* fill field with asterisks.                                    */
   if(num!=0)
   {
      memset(str,'*',ndig) ;
      str[ndig]=0 ;
      return ;
   }

   /* didn't fail, so add currency symbol if non-null, sign, and terminate string */
   if(sym!=0) *loc++=sym ;
   if(sign==-1) *loc++='-' ;
   *loc=0 ;

   /* now reverse the bytes in the string */
   len=strlen(str) ;
   for(i=0;i<=((len/2)-1);i++)
   {
      s=str[i] ;
      str[i]=str[len-1-i] ;
      str[len-1-i]=s ;
   } ;
} ;

/****************************************************************************
* Routne   : ctoi
* Gazintas : str = currency string to covnert to integer
* IOs      : num = conversion result
* Returns  : None
* Globals  : None
*
* This routine converts a currency string to an integer.  It is effectivly a
* slightly enhanced atoi that ignores a currency symbol and commas.  It also
* does a bit more error checking than atoi.
*
****************************************************************************/
uint8 ctoi_e(uint32* num,char* str)
{
   char* ptr ;

   *num = 0 ;
   ptr = str ;

   /* bypass any initial whitespace */
   while ((*ptr==' ')&&(*ptr!=0)) ptr++ ;

   while ( *ptr != 0 )
   {
      /* digit conversion */
      if ((*ptr>='0')&&(*ptr<='9'))
      {
         *num = *num * 10 + *ptr - '0' ;
         ptr++ ;
         continue ;
      } ;

      /* OK to ignore $'s and commas */
      if ((*ptr=='$')||(*ptr==','))
      {
         ptr++ ;
         continue ;
      } ;

      /* anything else, exit */
      break ;
   } ;

   /* if any trailing spaces (ONLY!), ignore them */
   while ((*ptr==' ')&&(*ptr!=0)) ptr++ ;

   /* if we ended at EOS, all went well, exit */
   if (*ptr==0) return FALSE;

   /* otherwise something wasn't well-formatted */
   return TRUE ;
} ;

/****************************************************************************
* Routne   : ctof
* Gazintas : str = currency string to covnert to float
* IOs      : num = conversion result
* Returns  : None
* Globals  : None
*
* This routine converts a currency string to a float.  It is effectivly a
* slightly enhanced atof that ignores a currency symbol and commas.  It also
* does a bit more error checking than atof.
*
* RIGHT NOW THIS ROUTINE DOES NOT SUPPORT DECIMALS!!!
*
* I don't need it for the first use, but need to make it more general purpose
* at some time.
****************************************************************************/
uint8 ctof_e(float* num,char* str)
{
   char* ptr ;

   *num = 0.0 ;
   ptr = str ;

   /* bypass any initial whitespace */
   while ((*ptr==' ')&&(*ptr!=0)) ptr++ ;

   while ( *ptr != 0 )
   {
      /* digit conversion */
      if ((*ptr>='0')&&(*ptr<='9'))
      {
         *num = *num * 10.0 + (float)(*ptr - '0') ;
         ptr++ ;
         continue ;
      } ;

      /* OK to ignore $'s and commas */
      if ((*ptr=='$')||(*ptr==','))
      {
         ptr++ ;
         continue ;
      } ;

      /* anything else, exit */
      break ;
   } ;

   /* if any trailing spaces (ONLY!), ignore them */
   while ((*ptr==' ')&&(*ptr!=0)) ptr++ ;

   /* if we ended at EOS, all went well, exit */
   if (*ptr==0) return FALSE;

   /* otherwise something wasn't well-formatted */
   return TRUE ;
} ;

/****************************************************************************
* Routne   : atoi_e
* Gazintas : str = integer string to covnert to integer
* IOs      : num = conversion result
* Returns  : None
* Globals  : None
*
* This routine converts a string integer to an integer.  It is essentially the
* atoi function, but with error checking for bad characters instead of just
* returning a partial value.
*
****************************************************************************/
uint8 atoi_e(uint32* num,char* str)
{
   char* ptr ;

   *num = 0 ;
   ptr = str ;

   /* bypass any initial whitespace */
   while ((*ptr==' ')&&(*ptr!=0)) ptr++ ;

   while ( *ptr != 0 )
   {
      /* digit conversion */
      if ((*ptr>='0')&&(*ptr<='9'))
      {
         *num = *num * 10 + *ptr - '0' ;
         ptr++ ;
         continue ;
      } ;

      /* anything else, exit */
      break ;
   } ;

   /* if any trailing spaces (ONLY!), ignore them */
   while ((*ptr==' ')&&(*ptr!=0)) ptr++ ;

   /* if we ended at EOS, all went well, exit */
   if (*ptr==0) return FALSE;

   /* otherwise something wasn't well-formatted */
   return TRUE ;
} ;

/****************************************************************************
* Routne   : atof_e
* Gazintas : str = currency string to covnert to float
* IOs      : num = conversion result
* Returns  : None
* Globals  : None
*
* This routine converts a string to a float.  It is essentially atof but with
* some error checking instead of just stopping at the first invalid character.
****************************************************************************/
uint8 atof_e(float* num,char* str)
{
   char* ptr ;
   float power = 1.0 ;

   *num = 0.0 ;
   ptr = str ;

   /* bypass any initial whitespace */
   while ((*ptr==' ')&&(*ptr!=0)) ptr++ ;

   while ( *ptr != 0 )
   {
      /* digit conversion */
      if ((*ptr>='0')&&(*ptr<='9'))
      {
         if (power==1.0)
         {
            *num = *num * 10.0 + (float)(*ptr - '0') ;
            ptr++ ;
            continue ;
         }
         else
         {
            *num = *num + power * (float)(*ptr - '0') ;
            power /= 10.0 ;
            ptr++ ;
            continue ;

         }
      } ;

      /* if we see a decimal point, start messing with the power */
      if (*ptr=='.')
      {
         power = 0.1 ;
         ptr++ ;
         continue ;
      } ;

      /* anything else, exit */
      break ;
   } ;

   /* if any trailing spaces (ONLY!), ignore them */
   while ((*ptr==' ')&&(*ptr!=0)) ptr++ ;

   /* if we ended at EOS, all went well, exit */
   if (*ptr==0) return FALSE;

   /* otherwise something wasn't well-formatted */
   return TRUE ;
} ;

