#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <math.h>
#include <time.h>
#include "def.h"
#include "Simulation.h"
#include "modes.h"
#include "params.h"
#include "sorts.h"
#include "reports.h"

/* These are Global Globals, but defined here                     */
SUMMARY summaries[N_MODES]        ; /* simulation summaries by mode */
RESULTS *results=NULL             ; /* results array                */
BATCH_SUMMARY_T batch_results[20] ; /* batch results array          */

FILE *flog    ; /* Log/Debug output file   */

/* Local routines */
void run_sim(float*,float*) ;
void yearly_init(YEARLY*) ;
void yearly_pre (YEARLY*) ;
void yearly_post(YEARLY*) ;

/****************************************************************************
* Routne   : run_batch
* Gazintas : None
* IOs      : None
* Globals  : BatchDat  : Batch-specific parameters
*            params    : General simulation parameters
*            summaries : summaries if final results for each mode
*            results   : Array of each simulations results
*
* This routine is the supervisory routine in charge of running a batch of
* simulations based on on the BatchDat structure.  I calls out the unique
* data for each full simuation, using the params structure for the more
* common and predictable stuff.  Basic idea is to run the simulation for 1 and
* 2 years both before and after the target year/age.  Spending catagories are
* adjusted by inflation as needed for the start year/age and the user supplies
* the predicted stash for eash starting year/age.  These simulations can be
* run both with and without SSI being assumed as well as with the baseline
* and an alternate CPI.  Hence up to four simulations for each year/age.
****************************************************************************/
int run_batch (void)
{
   int n,i,j,k ; /* General loop counters */
   int AltCPIs ; /* flag to use alt CPIs  */
   PARAMS Params_hold ;

   FILE *fbat ;
   fbat=fopen("Batch.txt","w") ;

   /* Start with clearing everything out */
   for (i=0;i<20;i++)
   {
      for (j=0;j<3;j++)
      {
         batch_results[i].mode[j]=0.0 ;
      } ;
   } ;

   /* We mess with the global parameter block quite a bit, so make a copy of  */
   /* it to use as baseline.                                                  */
   Params_hold=params ;

   /* Need a general flag that there is at least one alternate CPI */
   AltCPIs = BatchDat.UseAltCPI1|BatchDat.UseAltCPI2|BatchDat.UseAltCPI3 ;

   /* Set up counter limits for the two optional derivations */
   nbatchc=AltCPIs?2:1 ;
   nbatchs=BatchDat.BothSSIs?2:1 ;
   k=0 ;

   /* Three nested loops for batch mode.       */
   /*  - Loop 1: 5 years to try out (-2 to + 2 */
   /*  - Loop 2: Alternate CPIs (if any)       */
   /*  - Loop 3: SSI in/out (if requested)     */
   for (n=0;n<5;n++)
   {
      for (i=0;i<nbatchc;i++)
      {
         for (j=0;j<nbatchs;j++)
         {
            /* OK, have our indexes set up, time to go set up the data.       */
            /* Start with a clean copy of the baseline simulation parameters, */
            /* Then override the category CPIs on the second round if asked.  */
            params=Params_hold ;
            if ((BatchDat.UseAltCPI1)&&(i!=0)) params.cat1_expense_inflation=BatchDat.AltCPI ;
            if ((BatchDat.UseAltCPI2)&&(i!=0)) params.cat2_expense_inflation=BatchDat.AltCPI ;
            if ((BatchDat.UseAltCPI3)&&(i!=0)) params.cat3_expense_inflation=BatchDat.AltCPI ;

            /* Next, if Both SSI options are listed, force those to           */
            /* disabled and enabled.                                          */
            /* fancy logic leaves soc_sec_use  if BothSSIs not set, or sets   */
            /* alternately to false and true for each value of J (0,1).       */
            if (BatchDat.BothSSIs) params.soc_sec_use=(j!=0) ;

            /* Now that we have SSI and inflation set up, adjust the starting */
            /* expense catagories by -CPI^2, -CPI, 0, +CPI, and +CPI^2 based  */
            /* on the -2, -1, 0, +1, +2 starting year.  Adjust the starting   */
            /* year and age too.                                              */
            params.start_year +=(n-2) ;
            params.start_age  +=(n-2) ;
            params.start_amount = BatchDat.BStash[n] ;

            params.cat1_expense_amount_min *= pow(1.0+params.cat1_expense_inflation/100.0,(float)(n-2)) ;
            params.cat1_expense_amount_max *= pow(1.0+params.cat1_expense_inflation/100.0,(float)(n-2)) ;

            params.cat2_expense_amount_min *= pow(1.0+params.cat2_expense_inflation/100.0,(float)(n-2)) ;
            params.cat2_expense_amount_max *= pow(1.0+params.cat2_expense_inflation/100.0,(float)(n-2)) ;

            params.cat3_expense_amount_min *= pow(1.0+params.cat3_expense_inflation/100.0,(float)(n-2)) ;
            params.cat3_expense_amount_max *= pow(1.0+params.cat3_expense_inflation/100.0,(float)(n-2)) ;

            /* OK, all set up, run the simulation! */
            fprintf(fbat,"Retire in %4d, Age %2d, CPI %5.2f%%, %cSSI ",
                    params.start_year, params.start_age,params.cat1_expense_inflation,params.soc_sec_use?'+':'-') ;
            fprintf(fbat,"%8.0f %8.0f %8.0f %8.0f %8.0f %8.0f\n",
                         params.cat1_expense_amount_min,
                         params.cat1_expense_amount_max,
                         params.cat2_expense_amount_min,
                         params.cat2_expense_amount_max,
                         params.cat3_expense_amount_min,
                         params.cat3_expense_amount_max) ;
            run_sims() ;

            batch_results[k].mode[0]=summaries[0].successes*100/params.num_sims ;
            batch_results[k].mode[1]=summaries[1].successes*100/params.num_sims ;
            batch_results[k].mode[2]=summaries[2].successes*100/params.num_sims ;
            sprintf(batch_results[k].descr,"Retire in %4d, Age %2d, CPI %5.2f%%, %cSSI",
                    params.start_year, params.start_age,params.cat1_expense_inflation,params.soc_sec_use?'+':'-') ;

            /* on to next entry in the batch run */
            k++ ;

         }
      }
   }
   /* remember the number of batch runs and restore the original parameters */
   nbatch=k ;
   params=Params_hold ;

   fclose(fbat) ;

  return (0) ;
}

/****************************************************************************
* Routne   : run_sims
* Gazintas : None
* IOs      : None
* Globals  : params
*            summaries : summaries if final results for each mode
*            results : Array of each simulations results
*
* This routine is the supervisory routine in charge of running all of the
* requested simulations and tallying the results.  See run_sim to see what
* the simulation itself is all about.
****************************************************************************/
int run_sims (void)
{
   uint32 i,j                  ; /* general counters   */
   float estates[N_MODES]      ; /* Ending estates     */
   float avg_ror               ; /* Avg RoR from sim   */
   float tot_est[N_MODES]      ; /* estate totals      */

   /* If either logs or debug infor is requested, create the logfile */
   if (params.logs||params.debug)
   {
      if ((flog=fopen("Debug.txt","w")) == NULL)
      {
         perror("Debug file error:");
         return(1) ;
      } ;
   } ;


   /* Results initially set to NULL, so will be malloc first */
   results=realloc(results,sizeof(RESULTS)*params.num_sims) ;
   if (results==NULL)
   {
      if (params.logs||params.debug) fprintf(flog,"Malloc Error!\n") ;
      return (1) ;
   } ;

   /* Initialize the success count and min/max Leftovers (money after you die) */
   for (j=0;j<N_MODES;j++)
   {
      summaries[j].successes=0 ;
      summaries[j].min_left=0  ;
      summaries[j].max_left=0  ;
      summaries[j].mean_left=0 ;
      tot_est[j] = 0.0 ;
   } ;

   /* now for the main simulation loop! */
   for (i=0;i<params.num_sims;i++)
   {
      /* run a simulation, the estates array will contain the left-overs at */
      /* end of the simulation for each spending mode.  avg_ror is the      */
      /* average rate of returns over the simulation.                       */
      run_sim(estates,&avg_ror) ;

      /* save the averaged ror for the simulation */
      results[i].avg_ror  = avg_ror ;

      /* save the results for each spending mode */
      for (j=0;j<N_MODES;j++)
      {
         results[i].estate[j] = estates[j] ;

         /* total the estates so I can get the average */
         tot_est[j]+=estates[j] ;

         /* if there is money left over, it is a success, otherwise a failure */
         if (estates[j]>0.0) summaries[j].successes++ ;

         /* also track the min and max leftovers for each spending mode */
         /* I seed min/max unconditionally on the first iteration       */
         if ((i==0)||(estates[j]<summaries[j].min_left)) summaries[j].min_left=estates[j] ;
         if ((i==0)||(estates[j]>summaries[j].max_left)) summaries[j].max_left=estates[j] ;
      } ;
   } ;

   /* get the average estates by dividing the totals by the count */
   for (j=0;j<N_MODES;j++)
   {
      summaries[j].avg_left=tot_est[j]/params.num_sims ;
   } ;

   /* close debug file and exit */
   /* If either logs or debug infor were requested, close the logfile */
   if (params.logs||params.debug) fclose(flog) ;

   return (0) ;

} ;

/****************************************************************************
* Routne   : run_sim
* Gazintas : None
* IOs      : None
* Returns  : estates[] - leftover money if any at EOL for each simulation
* Globals  : params
*
* This routine is the engine to actually run a 1-year retirement simulation.  It
* sets up the intitial states for each of the modes and picks the same random
* rate of return for each simulation.  Picking the same random RoRs for each
* simulation is important so I can see how each mode handles the same inputs.
*
* This routine just sets up the framework, the math on what to spend is handled
* by called routines for each mode.  Similarly, the math for the actual years
* desired expenses and randomized rates of return are computed in the yearly
* helper routines.
*
* It is important to run the differente spending modes against the same set of
* random RoR's, that way any differences in outcome can be accurately attributed
* to spending modes, not different random RoR sequeces.
*
* Spending mode 1 uses the max value for each catagory always
* Spending mode 2 decreases spending to be <= the years RoR or min value.
* Spending mode 3 tries to maintain a spending pool of 3 years expenses.
*
* This routine returnes the "leftover money" at the end year for each mode.
* A positive value counts as a success while a negative amount is a failure.
*
****************************************************************************/
void run_sim (float estates[N_MODES], float* avg_ror)
{
   uint32 i                            ; /* basic counter                    */
   SIMD sim[N_MODES]                  ; /* our simulation tracker           */
   YEARLY yearly                      ; /* yearly stats                     */

   for (i=0;i<N_MODES;i++)
   {
      sim[i].start    = 0              ; /* Safe values   */
      sim[i].final=params.start_amount ; /* Initial stash */
      sim[i].rord     = 0              ; /* Safe values   */
      sim[i].spend1act= 0              ; /* Safe values   */
      sim[i].spend2act= 0              ; /* Safe values   */
      sim[i].spend3act= 0              ; /* Safe values   */
      sim[i].cache    = 0              ; /* Safe values   */
   } ;

   /* initialize yearly stuff */
   yearly_init (&yearly) ;

   /* For Mode 2, we maintain an expenses cache that we try to maintain at 3  */
   /* years spending money.  Seed that cache with 3 years cash.               */
   /* For this I don't mess with anticipating inflation.                      */
   sim[2].cache = (yearly.cat1max+yearly.cat2max+yearly.cat3max)*M2Years ;
   sim[2].final-=sim[2].cache ;

   /* if logs is on, show headers for yearly status */
   if (params.logs)
   {
      fprintf (flog,"                 ") ;
      fprintf (flog,"========================SPENDING MODE 0======================= ") ;
      fprintf (flog,"========================SPENDING MODE 1======================= ") ;
      fprintf (flog,"========================SPENDING MODE 2=======================\n") ;
      fprintf (flog,"                 ") ;
      fprintf (flog,"============================================================== ") ;
      fprintf (flog,"============================================================== ") ;
      fprintf (flog,"==============================================================\n") ;
      fprintf (flog,"year  age   rorp ") ;
      fprintf (flog,"   start     rord    cache %8.8s %8.8s %8.8s    final ",params.cat1_expense_name,params.cat2_expense_name,params.cat3_expense_name) ;
      fprintf (flog,"   start     rord    cache %8.8s %8.8s %8.8s    final ",params.cat1_expense_name,params.cat2_expense_name,params.cat3_expense_name) ;
      fprintf (flog,"   start     rord    cache %8.8s %8.8s %8.8s    final ",params.cat1_expense_name,params.cat2_expense_name,params.cat3_expense_name) ;
      fprintf (flog," SoC_Sec\n") ;
      fprintf (flog,"==== ==== ====== ") ;
      fprintf (flog,"======== ======== ======== ======== ======== ======== ======== ") ;
      fprintf (flog,"======== ======== ======== ======== ======== ======== ======== ") ;
      fprintf (flog,"======== ======== ======== ======== ======== ======== ======== ") ;
      fprintf (flog,"========\n") ;
   } ;

   /* loop for all the years in the planned retirement */
   for (yearly.age=params.start_age;yearly.age<=params.end_age;yearly.age++)
   {
      /* do the pre-mode yearly calculations */
      yearly_pre (&yearly) ;

      /* With the random RoR go do the math for the various theories */
      Mode0(&sim[0],&yearly) ;
      Mode1(&sim[1],&yearly) ;
      Mode2(&sim[2],&yearly) ;

      /* if logs are turned on, show all this data */
      if (params.logs)
      {
         fprintf (flog,"%4d %4d %6.2f ",yearly.year,yearly.age,yearly.rorp) ;
         for (i=0;i<N_MODES;i++)
         {
            fprintf (flog,"%8.0f %8.0f %8.0f %8.0f %8.0f %8.0f %8.0f ",sim[i].start,sim[i].rord,sim[i].cache,sim[i].spend1act,sim[i].spend2act,sim[i].spend3act,sim[i].final) ;
         } ;
         fprintf (flog,"%8.0f \n",yearly.soc_sec_adder) ;
      } ;

      /* do the post-mode yearly calculations */
      yearly_post(&yearly) ;

   } ; /* do all the years */

   /* we've summed up the RoR's so far, so divide to get the average RoR */
   *avg_ror=yearly.rort/(params.end_age-params.start_age+1) ;

   /* return the ending estate values */
   for (i=0;i<N_MODES;i++)
   {
      estates[i]=sim[i].final ;
   } ;
} ;

/****************************************************************************
* Routne   : yearly_init
* Gazintas : None
* IOs      : yearly
* Returns  : None
* Globals  : params
*
* This routine initializes all the spending data for a years worth of simulation.
* It sets up the initial values for the year, social security, and the expected
* range of RoRs.  It also zeros out the accumulated RoR so we can get the
* average at the end.
*
****************************************************************************/
void yearly_init (YEARLY* yearly)
{
   /* initialize yearly stuff */
   yearly->year = params.start_year               ; /* initial year               */
   yearly->soc_sec_adder=0.0                      ; /* set adder to 0             */

   yearly->cat1max=params.cat1_expense_amount_max ; /* initial spend (category 1) */
   yearly->cat2max=params.cat2_expense_amount_max ; /* initial spend (category 2) */
   yearly->cat3max=params.cat3_expense_amount_max ; /* initial spend (category 3) */

   yearly->cat1min=params.cat1_expense_amount_min ; /* minimum spend (category 1) */
   yearly->cat2min=params.cat2_expense_amount_min ; /* minimum spend (category 2) */
   yearly->cat3min=params.cat3_expense_amount_min ; /* minimum spend (category 3) */

   yearly->ror_avg = params.mkt_return_avg        ; /* RoR at start of retirement */
   yearly->ror_dev = params.mkt_return_range      ; /* Ror deviation              */

   yearly->rort = 0.0                             ; /* zero out totalized RoR     */

} ;

/****************************************************************************
* Routne   : yearly_pre
* Gazintas : None
* IOs      : yearly
* Returns  : None
* Globals  : params
*
* This routine does the yearly spending computations that happen before we do
* the computations based on mode.  This includes updating the RoRs on the
* appropriate years, computing the actual years random RoR, and updating
* Social Security if applicable.  It also sums up the RoRs for later avraging.
*
****************************************************************************/
void yearly_pre (YEARLY* yearly)
{
   /* if we get to one of the ages where the rates of return change, change them! */
   if (yearly->age==params.age_ror_phase2)
   {
      yearly->ror_avg = params.mkt_return_avg2    ; /* Phase 2 returns  */
      yearly->ror_dev = params.mkt_return_range2  ; /* Phase 2 range    */
   }
   else if (yearly->age==params.age_ror_phase3)
   {
      yearly->ror_avg = params.mkt_return_avg3    ; /* Phase 3 returns */
      yearly->ror_dev = params.mkt_return_range3  ; /* Phase 3 range   */
   }

   /* Get a random rate of return based on the above parameteric data */
   yearly->rorp=yearly->ror_avg+yearly->ror_dev-yearly->ror_dev*(float)rand()/(float)RAND_MAX*2.0 ;

   /* If we are getting social security and the right age, set the adder amount */
   if (params.soc_sec_use&&(yearly->age==params.soc_sec_age_start))
   {
      yearly->soc_sec_adder=params.soc_sec_amt ;
   }

   /* sum up the return percentages so I can figger out the average at the end */
   yearly->rort+=yearly->rorp ;
} ;

/****************************************************************************
* Routne   : yearly_post
* Gazintas : None
* IOs      : yearly
* Returns  : None
* Globals  : params
*
* This routine does the follow-up math on the yearly stats after the modal
* math is done.  This includes updating each spending category and Social
* Security by their inflation rates and incrementing the year.
*
****************************************************************************/
void yearly_post (YEARLY* yearly)
{
      /* increase max spending in each catagory by it's inflation rate */
      yearly->cat1max+=yearly->cat1max*params.cat1_expense_inflation/100.0 ;
      yearly->cat2max+=yearly->cat2max*params.cat2_expense_inflation/100.0 ;
      yearly->cat3max+=yearly->cat3max*params.cat3_expense_inflation/100.0 ;

      /* increase minimum spends by inflation too */
      yearly->cat1min+=yearly->cat1min*params.cat1_expense_inflation/100.0 ;
      yearly->cat2min+=yearly->cat2min*params.cat2_expense_inflation/100.0 ;
      yearly->cat3min+=yearly->cat3min*params.cat3_expense_inflation/100.0 ;

      /* inflate Social Security (if non-zero) */
      yearly->soc_sec_adder*=(1+params.soc_sec_inflation/100.0) ;

      /* increment the year */
      yearly->year++ ;
} ;

/****************************************************************************
* Routne   : sims_done
* Gazintas : None
* IOs      : None
* Globals  : results : Array of each simulations results
*
* This is just a clean-up routine after all simulations are complete.
* all it does is free up the malloc'd results.
****************************************************************************/
void sims_done (void)
{
   free(results) ;
} ;

