#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <math.h>
#include <time.h>
#include "def.h"
#include "cmdfile.h"
#include "MonteCarlo.h"
#include "modes.h"
#include "params.h"
#include "sorts.h"
#include "reports.h"

/* Local Globals */
FILE *fbatch  ; /* the batch control file */
FILE *fsumm   ; /* the batch summary file */
FILE *fcmd    ; /* the command file       */
FILE *fout    ; /* the Text output file   */
FILE *fcsv    ; /* the CSV output file    */

/* Local routines */
int  full_sim (char,char*,char*,SUMMARY*,char*) ;
void run_sims (SUMMARY*,RESULTS*) ;
void run_sim(float*,float*) ;
void yearly_init(YEARLY*) ;
void yearly_pre (YEARLY*) ;
void yearly_post(YEARLY*) ;

/******************************************************************************/
/* program MonteCarlo.c                                                       */
/*                                                                            */
/* This is my version of a Monte Carlo simulation for retirement planning.    */
/* I'm writing my own so I can fiddle with it as much as I want, but also     */
/* because I want to do more than what I can find other tools do.             */
/* The tools I can find assume all expense are fixed (except inflationary)    */
/* and the only thing that changes is rate of return.                         */
/* What I want to do is vary expenses too, based on rate of return.  This     */
/* follows the reality that in a bad market, you'd cut back on expenses.      */
/* I have other ideas too, but starting with this for now.                    */
/* Written for the gcc compiler in MinGW, though nothing should be special    */
/* about it.                                                                  */
/*                                                                            */
/* Rev 1.xx Intial codings (not really tracked)                               */
/* Rev 2.00 Major structural changes to allow >2 simulation modes             */
/*          Added floating lving expense buffer simulation                    */
/* Rev 2.10 Added batch support.                                              */
/******************************************************************************/
int main(int argc, char* argv[])
{
   SUMMARY summaries[N_MODES]  ; /* summaries by mode        */
   time_t tseed                ; /* time-based seed value    */
   int    rval                 ; /* return value             */
   char   line[81]             ; /* batch input line         */
   char*  param1               ; /* batch parameter pointers */
   char*  param2               ; /* batch parameter pointers */
   char*  param3               ; /* batch parameter pointers */
   int    simcount=0           ; /* batch simulation count   */
   int    nfiles               ; /* parameter file count     */
   int    i                    ; /* counter                  */
   float  percent              ; /* percentage               */

   printf("Welcome to the Monte Carlo retirement simulator! Rev 2.10\n") ;
   printf("The multi-mode retirement simulator.\n") ;

   /* make sure there is a parameter file specified and error out if not */
   if ((argc!=2)&&(argc!=3))
   {
      printf ("Usage: MonteCarlo paramfile1 <paramfile2>\n") ;
      printf ("  The first parameter file is mandatory.\n") ;
      printf ("  The secod is optional and duplicated entries will override first file.\n") ;
      printf ("    See example.txt for an example file.\n") ;
      printf (" If paramfile1 is 'batch.txt', this invokes batch mode.\n") ;
      exit(1) ;
   } ;

   /* seed the random number generator */
   /* seed only changes once/second    */
   srand((int)time(&tseed)) ;

   /* see if we are in one-time run mode (paramfile1!=batch.txt) */
   if (strncmp(argv[1],"batch.txt",9))
   {
      /* paramfile1 isn't batch.txt */
      rval=full_sim(argc,argv[1],argv[2],summaries,NULL) ;
   }
   else
   {
      /*paramfile1 is batch.txt */

      /* open up the batch control file */
      printf("\nOpening batch control file \"batch.txt\"...\n") ;
      if ((fbatch=fopen("batch.txt","r")) == NULL)
      {
          perror("Batch file error:");
          return(1) ;
      } ;

      /* open up the batch summary file */
      printf("\nOpening batch summar file \"summary.csv\"...\n") ;
      if ((fsumm=fopen("summary.csv","w")) == NULL)
      {
          perror("Batch summary file error:");
          return(1) ;
      } ;

      /* a header line for the summary file */
      fprintf(fsumm,"Simulation ID,") ;
      for (i=0;i<N_MODES;i++)
      {
         fprintf(fsumm,"Mode %d,",i) ;
      }
      fprintf(fsumm,"\n") ;

      /* Start reading lines from the batch file and feed the data to the     */
      /* simulator.                                                           */
      while (!feof(fbatch))
      {
         line[0]=0 ;
         fgets(line,80,fbatch)    ; /* get the line */
         if (strlen(line)<5)        /* ignore short lines */
         {
            fprintf (fsumm,"\n")  ; /* translate to a blank line in summary   */
            continue              ;
         } ;
         line[strlen(line)-1]=0   ; /* kill the EOL */
//         printf(">%s<\n",line)  ; /* print it     */

         /* Parse the line, first parameter is the quoted string for the      */
         /* simulation description, second is the first control file name,    */
         /* third is the optional second file name.                           */
         /* use the existing line, just store pointers and null-terminate.    */

         param1=line ; /* description always starts with first character      */
         param2=strchr(line+1,'\"') ; /* look for ending quote                */
         if (param2==NULL)  /* if none, error                                 */
         {
            printf("Error in bach file, no closing quote for sim ID\n") ;
            return(1) ;
         }
         param2++     ; /* character after the quote...                       */
         param2[0]= 0 ; /* is the end of the first parameter                  */
         param2++     ; /* param2 is now pointer to first control file        */

         param3=strchr(param2,' ') ; /* look for space between file names     */
         if (param3==NULL) /* only process if there is a third parameter      */
         {
            nfiles = 1 ; /* set the file count                                */
         }
         else
         {
            param3[0]= 0 ; /* is the end of the first parameter               */
            param3++     ; /* param3 is now pointer to first control file     */
            nfiles = 2   ; /* Set the file count                              */
         }

         printf ("Batch run: %s %s ",param1,param2) ;
         if (nfiles==2) printf ("%s ",param3) ;
         printf ("\n") ;

         /* finally have everything, so go run the simulation.                */
         rval=full_sim(nfiles+1,param2,param3,summaries,param1) ;
         fprintf(fsumm,"%s,",param1) ;
         for (i=0;i<N_MODES;i++)
         {
            percent = (float)summaries[i].successes/(float)params.num_sims*100.0 ;
            fprintf(fsumm,"%2.0f%%,",percent) ;
         }
         fprintf(fsumm,"\n") ;

         simcount++ ;
      } ;

      rval = 1 ;
      printf ("%d batch simulations performed\n",simcount) ;

   } ;

   return(rval) ;
} ;

/****************************************************************************
* Routne   : full_sim
* Gazintas : file count = number of control files
*            file1      = control file 1
*            file2      = control file 2
*            batchname  = batch flag and batch output name
* IOs      : summaries  = simulation results
* Returns  : pass/fail code
* Globals  : params
*
* This routine is top level simulation routine that reads the parameter files,
* invokes the full simulation, and reports the results.  For a non-batch run,
* this routine gets executed only once.  For a batch run, it gets run each
* entry of the batch control file.  The batchname is null for single runs.
* For batch runs batchname will become the output file name and the non-null
* value also supresses all of the screen outputs.
****************************************************************************/
int full_sim(char fcount, char* file1, char* file2, SUMMARY summaries[],char* batchname)
{
   RESULTS *results            ; /* results array         */
   uint8 error = FALSE         ; /* general error flag    */
   char foutname[70]           ; /* output filename (64 for base + 4 byte extension */
   int batchmode               ; /* flag to indicate batch mode               */

   batchmode = (batchname!=NULL) ;

   /* set our "first time" flag for parameter setting.  This ensures          */
   /* defaults are loaded on the first (and only) call to read_params.        */
   params.first=TRUE ;

   /* go read the paramter file.  This has all the user data and parameters   */
   /* to drive the simulation.  See read_params for details.                  */
   error=read_params (file1,batchmode) ;

   /* if there is a second file, read it next */
   if (fcount==3)
   {
   error|=read_params (file2,batchmode) ;
   }

   /* create and open the report file, abort if fail */
   if (batchmode)
   {
      /* copy ID, but remove quotes */
      strcpy(foutname,batchname+1) ;
      foutname[strlen(foutname)-1]=0;
      strcat (foutname,".txt") ;
   }
   else
   {
      strcpy (foutname,params.outfile) ;
      strcat (foutname,".txt") ;
   }

   if ((fout=fopen(foutname,"w")) == NULL)
   {
       perror("Report file error:");
       return(1) ;
   }

   /* write out the input parameters to the report file (even if errors) */
   write_params() ;

   /* if there were input errors, say so, and abort */
   if (error)
   {
      printf ("Errors in the input file, see %s.txt for results./n",params.outfile) ;
      return (1) ;
   }

   /* I store the basic results of each simulation.  As the number of       */
   /* simulations is a user input, I dynamically allocate the memory for    */
   /* the results.                                                          */
   results=malloc(sizeof(RESULTS)*params.num_sims) ;
   if (results==NULL)
   {
      printf("Malloc Error!\n") ;
      return (1) ;
   } ;

   /* go run all the simulations */
   run_sims(summaries,results) ;

   /* If requested, print out the various reports */
   report_percentile(results)                 ;
   report_histogram(summaries,results)        ;
   report_summaries(summaries,results,fout)   ;
   if (!batchmode) report_summaries(summaries,results,stdout) ;

   /* if we got this far, no errors, return "success" */
   fclose (fout) ;
   return (0) ;
} ;

/****************************************************************************
* Routne   : run_sims
* Gazintas : None
* IOs      : None
* Returns  : summaries : summaries if final results for each mode
*            results : Array of each simulations results
* Globals  : params
*
* This routine is the supervisory routine in charge of running all of the
* requested simulations and tallying the results.  See run_sim to see what
* the simulation itself is all about.
****************************************************************************/
void run_sims (SUMMARY* summaries,RESULTS* results)
{
   uint32 i,j                   ; /* general counters   */
   float estates[N_MODES]      ; /* Ending estates     */
   float avg_ror               ; /* Avg RoR from sim   */
   float tot_est[N_MODES]      ; /* estate totals      */

   /* Initialize the success count and min/max Leftovers (money after you die) */
   for (j=0;j<N_MODES;j++)
   {
      summaries[j].successes=0 ;
      summaries[j].min_left=0  ;
      summaries[j].max_left=0  ;
      summaries[j].mean_left=0 ;
      tot_est[j] = 0.0 ;
   } ;

   /* now for the main simulation loop! */
   for (i=0;i<params.num_sims;i++)
   {
      /* run a simulation, the estates array will contain the left-overs at */
      /* end of the simulation for each spending mode.  avg_ror is the      */
      /* average rate of returns over the simulation.                       */
      run_sim(estates,&avg_ror) ;

      /* save the averaged ror for the simulation */
      results[i].avg_ror  = avg_ror ;

      /* save the results for each spending mode */
      for (j=0;j<N_MODES;j++)
      {
         results[i].estate[j] = estates[j] ;

         /* total the estates so I can get the average */
         tot_est[j]+=estates[j] ;

         /* if there is money left over, it is a success, otherwise a failure */
         if (estates[j]>0.0) summaries[j].successes++ ;

         /* also track the min and max leftovers for each spending mode */
         /* I seed min/max unconditionally on the first iteration       */
         if ((i==0)||(estates[j]<summaries[j].min_left)) summaries[j].min_left=estates[j] ;
         if ((i==0)||(estates[j]>summaries[j].max_left)) summaries[j].max_left=estates[j] ;
      } ;
   } ;

   /* get the average estates by dividing the totals by the count */
   for (j=0;j<N_MODES;j++)
   {
      summaries[j].avg_left=tot_est[j]/params.num_sims ;
   } ;

} ;

/****************************************************************************
* Routne   : run_sim
* Gazintas : None
* IOs      : None
* Returns  : estates[] - leftover money if any at EOL for each simulation
* Globals  : params
*
* This routine is the engine to actually run a 1-year retirement simulation.  It
* sets up the intitial states for each of the modes and picks the same random
* rate of return for each simulation.  Picking the same random RoRs for each
* simulation is important so I can see how each mode handles the same inputs.
*
* This routine just sets up the framework, the math on what to spend is handled
* by called routines for each mode.  Similarly, the math for the actual years
* desired expenses and randomized rates of return are computed in the yearly
* helper routines.
*
* It is important to run the differente spending modes against the same set of
* random RoR's, that way any differences in outcome can be accurately attributed
* to spending modes, not different random RoR sequeces.
*
* Spending mode 1 uses the max value for each catagory always
* Spending mode 2 decreases spending to be <= the years RoR or min value.
* Spending mode 3 tries to maintain a spending pool of 3 years expenses.
*
* This routine returnes the "leftover money" at the end year for each mode.
* A positive value counts as a success while a negative amount is a failure.
*
****************************************************************************/
void run_sim (float estates[N_MODES], float* avg_ror)
{
   uint32 i                            ; /* basic counter                    */
   SIMD sim[N_MODES]                  ; /* our simulation tracker           */
   YEARLY yearly                      ; /* yearly stats                     */

   for (i=0;i<N_MODES;i++)
   {
      sim[i].start    = 0              ; /* Safe values   */
      sim[i].final=params.start_amount ; /* Initial stash */
      sim[i].rord     = 0              ; /* Safe values   */
      sim[i].spend1act= 0              ; /* Safe values   */
      sim[i].spend2act= 0              ; /* Safe values   */
      sim[i].spend3act= 0              ; /* Safe values   */
      sim[i].cache    = 0              ; /* Safe values   */
   } ;

   /* initialize yearly stuff */
   yearly_init (&yearly) ;

   /* For Mode 2, we maintain an expenses cache that we try to maintain at 3  */
   /* years spending money.  Seed that cache with 3 years cash.               */
   /* For this I don't mess with anticipating inflation.                      */
   sim[2].cache = (yearly.cat1max+yearly.cat2max+yearly.cat3max)*M2Years ;
   sim[2].final-=sim[2].cache ;

   /* if logs is on, show headers for yearly status */
   if (params.logs)
   {
      fprintf (fout,"                 ") ;
      fprintf (fout,"========================SPENDING MODE 0======================= ") ;
      fprintf (fout,"========================SPENDING MODE 1======================= ") ;
      fprintf (fout,"========================SPENDING MODE 2=======================\n") ;
      fprintf (fout,"                 ") ;
      fprintf (fout,"============================================================== ") ;
      fprintf (fout,"============================================================== ") ;
      fprintf (fout,"==============================================================\n") ;
      fprintf (fout,"year  age   rorp ") ;
      fprintf (fout,"   start     rord    cache %8.8s %8.8s %8.8s    final ",params.cat1_expense_name,params.cat2_expense_name,params.cat3_expense_name) ;
      fprintf (fout,"   start     rord    cache %8.8s %8.8s %8.8s    final ",params.cat1_expense_name,params.cat2_expense_name,params.cat3_expense_name) ;
      fprintf (fout,"   start     rord    cache %8.8s %8.8s %8.8s    final ",params.cat1_expense_name,params.cat2_expense_name,params.cat3_expense_name) ;
      fprintf (fout," SoC_Sec\n") ;
      fprintf (fout,"==== ==== ====== ") ;
      fprintf (fout,"======== ======== ======== ======== ======== ======== ======== ") ;
      fprintf (fout,"======== ======== ======== ======== ======== ======== ======== ") ;
      fprintf (fout,"======== ======== ======== ======== ======== ======== ======== ") ;
      fprintf (fout,"========\n") ;
   } ;

   /* loop for all the years in the planned retirement */
   for (yearly.age=params.start_age;yearly.age<=params.end_age;yearly.age++)
   {
      /* do the pre-mode yearly calculations */
      yearly_pre (&yearly) ;

      /* With the random RoR go do the math for the various theories */
      Mode0(&sim[0],&yearly) ;
      Mode1(&sim[1],&yearly) ;
      Mode2(&sim[2],&yearly) ;

      /* if logs are turned on, show all this data */
      if (params.logs)
      {
         fprintf (fout,"%4d %4d %6.2f ",yearly.year,yearly.age,yearly.rorp) ;
         for (i=0;i<N_MODES;i++)
         {
            fprintf (fout,"%8.0f %8.0f %8.0f %8.0f %8.0f %8.0f %8.0f ",sim[i].start,sim[i].rord,sim[i].cache,sim[i].spend1act,sim[i].spend2act,sim[i].spend3act,sim[i].final) ;
         } ;
         fprintf (fout,"%8.0f \n",yearly.soc_sec_adder) ;
      } ;

      /* do the post-mode yearly calculations */
      yearly_post(&yearly) ;

   } ; /* do all the years */

   /* we've summed up the RoR's so far, so divide to get the average RoR */
   *avg_ror=yearly.rort/(params.end_age-params.start_age+1) ;

   /* return the ending estate values */
   for (i=0;i<N_MODES;i++)
   {
      estates[i]=sim[i].final ;
   } ;
} ;

/****************************************************************************
* Routne   : yearly_init
* Gazintas : None
* IOs      : yearly
* Returns  : None
* Globals  : params
*
* This routine initializes all the spending data for a years worth of simulation.
* It sets up the initial values for the year, social security, and the expected
* range of RoRs.  It also zeros out the accumulated RoR so we can get the
* average at the end.
*
****************************************************************************/
void yearly_init (YEARLY* yearly)
{
   /* initialize yearly stuff */
   yearly->year = params.start_year               ; /* initial year               */
   yearly->soc_sec_adder=0.0                      ; /* set adder to 0             */

   yearly->cat1max=params.cat1_expense_amount_max ; /* initial spend (category 1) */
   yearly->cat2max=params.cat2_expense_amount_max ; /* initial spend (category 2) */
   yearly->cat3max=params.cat3_expense_amount_max ; /* initial spend (category 3) */

   yearly->cat1min=params.cat1_expense_amount_min ; /* minimum spend (category 1) */
   yearly->cat2min=params.cat2_expense_amount_min ; /* minimum spend (category 2) */
   yearly->cat3min=params.cat3_expense_amount_min ; /* minimum spend (category 3) */

   yearly->ror_avg = params.mkt_return_avg        ; /* RoR at start of retirement */
   yearly->ror_dev = params.mkt_return_range      ; /* Ror deviation              */

   yearly->rort = 0.0                             ; /* zero out totalized RoR     */

} ;

/****************************************************************************
* Routne   : yearly_pre
* Gazintas : None
* IOs      : yearly
* Returns  : None
* Globals  : params
*
* This routine does the yearly spending computations that happen before we do
* the computations based on mode.  This includes updating the RoRs on the
* appropriate years, computing the actual years random RoR, and updating
* Social Security if applicable.  It also sums up the RoRs for later avraging.
*
****************************************************************************/
void yearly_pre (YEARLY* yearly)
{
   /* if we get to one of the ages where the rates of return change, change them! */
   if (yearly->age==params.age_ror_phase2)
   {
      yearly->ror_avg = params.mkt_return_avg2    ; /* Phase 2 returns  */
      yearly->ror_dev = params.mkt_return_range2  ; /* Phase 2 range    */
   }
   else if (yearly->age==params.age_ror_phase3)
   {
      yearly->ror_avg = params.mkt_return_avg3    ; /* Phase 3 returns */
      yearly->ror_dev = params.mkt_return_range3  ; /* Phase 3 range   */
   }

   /* Get a random rate of return based on the above parameteric data */
   yearly->rorp=yearly->ror_avg+yearly->ror_dev-yearly->ror_dev*(float)rand()/(float)RAND_MAX*2.0 ;

   /* If we are getting social security and the right age, set the adder amount */
   if (params.soc_sec_use&&(yearly->age==params.soc_sec_age_start))
   {
      yearly->soc_sec_adder=params.soc_sec_amt ;
   }

   /* sum up the return percentages so I can figger out the average at the end */
   yearly->rort+=yearly->rorp ;
} ;

/****************************************************************************
* Routne   : yearly_post
* Gazintas : None
* IOs      : yearly
* Returns  : None
* Globals  : params
*
* This routine does the follow-up math on the yearly stats after the modal
* math is done.  This includes updating each spending category and Social
* Security by their inflation rates and incrementing the year.
*
****************************************************************************/
void yearly_post (YEARLY* yearly)
{
      /* increase max spending in each catagory by it's inflation rate */
      yearly->cat1max+=yearly->cat1max*params.cat1_expense_inflation/100.0 ;
      yearly->cat2max+=yearly->cat1max*params.cat2_expense_inflation/100.0 ;
      yearly->cat3max+=yearly->cat1max*params.cat3_expense_inflation/100.0 ;

      /* increase minimum spends by inflation too */
      yearly->cat1min+=yearly->cat1min*params.cat1_expense_inflation/100.0 ;
      yearly->cat2min+=yearly->cat1min*params.cat2_expense_inflation/100.0 ;
      yearly->cat3min+=yearly->cat1min*params.cat3_expense_inflation/100.0 ;

      /* inflate Social Security (if non-zero) */
      yearly->soc_sec_adder*=(1+params.soc_sec_inflation/100.0) ;

      /* increment the year */
      yearly->year++ ;
} ;
